/*
   
   CardGame3D Engine
   Copyright 2005 - Meusesoft
   
   Version 0.1: November 2005 - 
   
   
   
   Module Renderer
   
   Contains the code for rendering the environment of the game in OpenGL   
   
*/
#include "DataStructures.h"
#include "System.h"
#include "World.h"
#include "Renderer.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <sstream>

// Constructor and destructor

cRenderer::cRenderer(cSystem* poSystem, cWorld* poWorld) {

    oSystem = poSystem;
    oWorld = poWorld;

	oMesh = new cMesh(oSystem, oWorld);
	oRoom = new cMesh(oSystem, oWorld);

    //Load textures and fonts
    r_LoadTextures();

	//Load shaders
	r_LoadShaders();

    //Initialise the data structure
    r_RendererInitStructures();

    //Initialise OpenGL State
	glShadeModel(GL_FLAT);
	glShadeModel(GL_SMOOTH);
    //glEnable(GL_MULTISAMPLE_EXT);
	
    glHint(GL_MULTISAMPLE_FILTER_HINT_NV, GL_FASTEST);
    glHint(GL_MULTISAMPLE_FILTER_HINT_NV, GL_NICEST);
    
    guDisk = gluNewQuadric();
	gluQuadricNormals(guDisk, GLU_SMOOTH);	
	gluQuadricTexture(guDisk, GL_TRUE);	
    gluQuadricDrawStyle(guDisk, GLU_FILL);
    gluQuadricOrientation(guDisk, GLU_INSIDE);
    }
     
cRenderer::~cRenderer() {
     
	SaveSettings();
	
	glDeleteTextures(100, oWorld->lTextures);
    delete[] oWorld->lTextures;

	r_ClearShaders();
    
    dFrameTickCount.clear();
    
    gluDeleteQuadric(guDisk);

	delete oMesh;
	delete oRoom;
     }

void cRenderer::LoadSettings() {

	oWorld->fEyeDistance = (float)oSystem->ReadInteger("Renderer", "EyeDistance", -9000) / 1000.f;	
	oWorld->fEyeRotation = (float)oSystem->ReadInteger("Renderer", "EyeRotation", -15000) / 1000.f;	
}


void cRenderer::SaveSettings() {

	oSystem->WriteInteger("Renderer", "EyeDistance", (int)oWorld->fEyeDistance * 1000);
	oSystem->WriteInteger("Renderer", "EyeRotation", (int)oWorld->fEyeRotation * 1000);
	}
		
// Events

//This event happens when the window size is changed
void cRenderer::onSize() {

     glViewport(0, 0, oSystem->iGameWindowWidth, oSystem->iGameWindowHeight);
     }

//This events happens on a mouse move while the left buttons is pressed. The users changes its perspective
//on the world by dragging the mouse     
void cRenderer::onRotateWorld(float pfDeltaX, float pfDeltaY) {

    GLint viewport[4];
    glGetIntegerv (GL_VIEWPORT, viewport);
    
    oWorld->fRotateWorldXAxis = -(pfDeltaY * 45.0f / (viewport[3] - viewport[1]));
    oWorld->fRotateWorldYAxis = -(pfDeltaX * 45.0f / (viewport[3] - viewport[1]));
    }


float cRenderer::TimeBetweenLastFrames() {
     
	float fTimePassed;
    
	fTimePassed = 1;

	if (dFrameTickCount.size()>1) {
		fTimePassed = (float)dFrameTickCount[0] - (float)dFrameTickCount[1];
	}

    return fTimePassed;
	}
     
//this function does some initialisation for the start of the game

void cRenderer::r_InitGame() {

	//empty for now
	}

void cRenderer::r_RendererInitStructures() {

    oWorld->MaterialConstants.no_mat[0] = 0.0f;
    oWorld->MaterialConstants.no_mat[1] = 0.0f;
    oWorld->MaterialConstants.no_mat[2] = 0.0f;
    oWorld->MaterialConstants.no_mat[3] = 1.0f;
    oWorld->MaterialConstants.mat_ambient[0] = 0.7f;
    oWorld->MaterialConstants.mat_ambient[1] = 0.7f;
    oWorld->MaterialConstants.mat_ambient[2] = 0.7f;
    oWorld->MaterialConstants.mat_ambient[3] = 1.0f;
    oWorld->MaterialConstants.mat_ambient_color[0] = 0.8f;
    oWorld->MaterialConstants.mat_ambient_color[1] = 0.8f;
    oWorld->MaterialConstants.mat_ambient_color[2] = 0.2f;
    oWorld->MaterialConstants.mat_ambient_color[3] = 1.0f;
    oWorld->MaterialConstants.mat_diffuse[0] = 0.1f;
    oWorld->MaterialConstants.mat_diffuse[1] = 0.5f;
    oWorld->MaterialConstants.mat_diffuse[2] = 0.8f;
    oWorld->MaterialConstants.mat_diffuse[3] = 1.0f;
    oWorld->MaterialConstants.mat_specular[0] = 1.0f;
    oWorld->MaterialConstants.mat_specular[1] = 1.0f;
    oWorld->MaterialConstants.mat_specular[2] = 1.0f;
    oWorld->MaterialConstants.mat_specular[3] = 1.0f;
    oWorld->MaterialConstants.no_shininess[0] =  0.0f;
    oWorld->MaterialConstants.low_shininess[0] =  5.0f;
    oWorld->MaterialConstants.high_shininess[0] = 100.0f;
    oWorld->MaterialConstants.mat_emission[0] = 0.3f;
    oWorld->MaterialConstants.mat_emission[1] = 0.2f;
    oWorld->MaterialConstants.mat_emission[2] = 0.2f;
    oWorld->MaterialConstants.mat_emission[3] = 0.0f;
    
    oWorld->RenderOptions.bRenderLines = false;
    oWorld->RenderOptions.bPositionEyeFarAway = false;
    oWorld->RenderOptions.bRenderEnvironment = true;
    oWorld->RenderOptions.bFullLight = false;
	oWorld->RenderOptions.bDrawCardLines = false;
    
    oWorld->lRoomDisplayList = 0;
    oWorld->lCardDisplayList = 0;
    oWorld->fTextureRepeat = 1.0f;

    float fRotateWorldXAxis = 0.0;
    float fRotateWorldYAxis = 0.0;
    
    strcpy(oText.cText,"");
    oText.iX = 10;
    oText.iY = 10;
	oText.cTextRect.iTop = 10;
	oText.cTextRect.iRight = 200;
	oText.cTextRect.iLeft = 10;
	oText.cTextRect.iBottom = 40;
    oText.vColor[0] = 255;
	oText.vColor[1] = 255;
	oText.vColor[2] = 255;
	oText.fTransparancy = 1.0f;
	oText.fSize = 30;
    oText.dStyle = styleSizeFixed;

	LoadSettings();        
}


void cRenderer::CrossProduct(vec_f pV1, vec_f pV2, vec_f pVcross) {
	pVcross[0] = pV1[1]*pV2[2] - pV1[2]*pV2[1];
	pVcross[1] = pV1[2]*pV2[0] - pV1[0]*pV2[2];
	pVcross[2] = pV1[0]*pV2[1] - pV1[1]*pV2[0];
}

void cRenderer::NormalizeVector(vec_f pVin, vec_f pVnormal) {
	
	float length;

	length = sqrt (pVin[0]*pVin[0] + pVin[1]*pVin[1] + pVin[2]*pVin[2]);
	if (length == 0)
	{
       length = 1;
	}

	pVnormal[0] = pVin[0]/length;
	pVnormal[1] = pVin[1]/length;
	pVnormal[2] = pVin[2]/length;
}

//this function initiates the extensions for the GLSL shaders
void cRenderer::r_InitShaders() {

	// This grabs a list of all the video card's extensions it supports
	char *szGLExtensions = (char*)glGetString(GL_EXTENSIONS);

	// Make sure find the GL_ARB_shader_objects extension so we can use shaders.
	if(!strstr(szGLExtensions, "GL_ARB_shader_objects"))
	{
		bShaders = false;
		//MessageBox(NULL, "GL_ARB_shader_objects extension not supported!", "Error", MB_OK);
		return;
	}

	// Make sure we support the GLSL shading language 1.0
	if(!strstr(szGLExtensions, "GL_ARB_shading_language_100"))
	{
		bShaders = false;
		//MessageBox(NULL, "GL_ARB_shading_language_100 extension not supported!", "Error", MB_OK);
		return;
    }

	// Now let's set all of our function pointers for our extension functions
	glCreateShaderObjectARB = (PFNGLCREATESHADEROBJECTARBPROC)wglGetProcAddress("glCreateShaderObjectARB");
	glShaderSourceARB = (PFNGLSHADERSOURCEARBPROC)wglGetProcAddress("glShaderSourceARB");
	glCompileShaderARB = (PFNGLCOMPILESHADERARBPROC)wglGetProcAddress("glCompileShaderARB");
	glCreateProgramObjectARB = (PFNGLCREATEPROGRAMOBJECTARBPROC)wglGetProcAddress("glCreateProgramObjectARB");
	glAttachObjectARB = (PFNGLATTACHOBJECTARBPROC)wglGetProcAddress("glAttachObjectARB");
	glLinkProgramARB = (PFNGLLINKPROGRAMARBPROC)wglGetProcAddress("glLinkProgramARB");
	glUseProgramObjectARB = (PFNGLUSEPROGRAMOBJECTARBPROC)wglGetProcAddress("glUseProgramObjectARB");
	glUniform1iARB = (PFNGLUNIFORM1IARBPROC)wglGetProcAddress("glUniform1iARB");
	glUniform1fARB = (PFNGLUNIFORM1FARBPROC)wglGetProcAddress("glUniform1fARB");
	glUniform2fARB = (PFNGLUNIFORM2FARBPROC)wglGetProcAddress("glUniform2fARB");
	glUniform3fARB = (PFNGLUNIFORM3FARBPROC)wglGetProcAddress("glUniform3fARB");
	glUniform4fARB = (PFNGLUNIFORM4FARBPROC)wglGetProcAddress("glUniform4fARB");
	glGetUniformLocationARB = (PFNGLGETUNIFORMLOCATIONARBPROC)wglGetProcAddress("glGetUniformLocationARB");
	glDetachObjectARB = (PFNGLDETACHOBJECTARBPROC)wglGetProcAddress("glDetachObjectARB");
	glDeleteObjectARB  = (PFNGLDELETEOBJECTARBPROC)wglGetProcAddress("glDeleteObjectARB");
	glGetObjectParameterivARB  = (PFNGLGETOBJECTPARAMETERIVARBPROC)wglGetProcAddress("glGetObjectParameterivARB");
	glGetInfoLogARB  = (PFNGLGETINFOLOGARBPROC)wglGetProcAddress("glGetInfoLogARB");
	glActiveTextureARB = (PFNGLACTIVETEXTUREARBPROC)wglGetProcAddress("glActiveTextureARB"); 		 		
	bShaders = true; 
	}
	
//this function is the base function for loading all shaders into memory
void cRenderer::r_LoadShaders() {

	r_InitShaders();

	//check if shaders are enabled
	if (!bShaders) return;

	r_LoadShader(s_table, "shaders\\shader_table.vert", "shaders\\shader_table.frag");
	r_LoadShader(s_cardselected, "shaders\\shader_card.vert", "shaders\\shader_card.frag");
	}

//This functions loads a single shader into memory. It returns the index of the
//shader its handle in the vector
long cRenderer::r_LoadShader(long plShaderId, char* pcVertexFilename, char* pcFragmentFilename) {

	if (!bShaders) return -1;

	GLhandleARB hVertexShader;
	GLhandleARB hFragmentShader;
	GLhandleARB hProgramObject;
	sShader oShader;

	//Load and compile the shader objects
	hVertexShader = r_LoadShaderObject(pcVertexFilename, GL_VERTEX_SHADER_ARB);
	hFragmentShader = r_LoadShaderObject(pcFragmentFilename, GL_FRAGMENT_SHADER_ARB);

	if (hVertexShader==NULL || hFragmentShader==NULL) {
		
		if (hVertexShader!=NULL) glDeleteObjectARB(hVertexShader);
		if (hFragmentShader!=NULL) glDeleteObjectARB(hFragmentShader);

		return -1;
		}
	
	//Create a program object to represent the shaders
	hProgramObject = glCreateProgramObjectARB();

	//Attachd the sahders to the program object
	glAttachObjectARB(hProgramObject, hVertexShader);
	glAttachObjectARB(hProgramObject, hFragmentShader);

	//Link the program object
	glLinkProgramARB(hProgramObject);

	//Put a shader structure in the oShaders vector.
	oShader.lShaderId = plShaderId;
	oShader.hProgramObject = hProgramObject;
	oShader.hFragmentShader = hFragmentShader;
	oShader.hVertexShader = hVertexShader;
	oShaders.push_back(oShader);
}

//This functions loads a single shader into memory. It returns the handle of the
//shader
GLhandleARB cRenderer::r_LoadShaderObject(char* pcFilename, GLenum shaderType) {

	if (!bShaders) return NULL;

	GLhandleARB hShader;
    int infologLength = 0;
    int charsWritten  = 0;
    char *infoLog;
	char* cShaderSource;
	
	//check the shadertype
	if (shaderType!=GL_VERTEX_SHADER_ARB && shaderType!=GL_FRAGMENT_SHADER_ARB) {
		return -1;
		}
	
	//Create the shader container
	hShader = glCreateShaderObjectARB(shaderType); 

	//Read in the shader
	cShaderSource = textFileRead(pcFilename);
	const char *ccShaderSource = cShaderSource;
	
	glShaderSourceARB(hShader, 1, &ccShaderSource,NULL);
	free(cShaderSource);

	//Compile the shader
	glCompileShaderARB(hShader);

	//Check the info log for erros
    glGetObjectParameterivARB(hShader, GL_OBJECT_INFO_LOG_LENGTH_ARB, &infologLength);
	
    if (infologLength > 1) {

		infoLog = (char *)malloc(infologLength);
		glGetInfoLogARB(hShader, infologLength, &charsWritten, infoLog);
			
		MessageBox(NULL, infoLog, "Error", MB_OK);
		
		free(infoLog);
		glDeleteObjectARB(hShader);
		hShader = NULL;
	    }

	//Return the handle of the compiled shader
	return hShader;
}

//This function enables the shader identified by plShaderId in OpenGl.
bool cRenderer::r_UseShader(long plShaderId) {

	bool bFound;
	long lIndex;

	if (!bShaders) return false;

	lIndex = oShaders.size()-1;
	bFound = false;

	while (lIndex>=0 && !bFound) {

		if (oShaders[lIndex].lShaderId == plShaderId) {

			glUseProgramObjectARB(oShaders[lIndex].hProgramObject);
			bFound = true;
			}
		lIndex--;
		}

	return bFound;
	 }

//This function turns off the current shader
void cRenderer::r_TurnOffShader() {

	if (bShaders) glUseProgramObjectARB(0);
	}


//This function detaches and releases all shaders in (OpenGL) memory.
void cRenderer::r_ClearShaders() {

	sShader oShader;

	for (unsigned long lIndex=0; lIndex<oShaders.size(); lIndex++) {

		oShader = oShaders[lIndex];

		//Detach and delete the vertex shader
		if(oShader.hVertexShader) {

			glDetachObjectARB(oShader.hProgramObject, oShader.hVertexShader);
			glDeleteObjectARB(oShader.hVertexShader);
			}

		//Detach and delete the fragment shader
		if(oShader.hFragmentShader) {
			
			glDetachObjectARB(oShader.hProgramObject, oShader.hFragmentShader);
			glDeleteObjectARB(oShader.hFragmentShader);
			}

		//Delete the program object
		if(oShader.hProgramObject) {

			glDeleteObjectARB(oShader.hProgramObject);
			}
		}
}
	
//this functions is the base function for loading all textures into memory

void cRenderer::r_LoadTextures() {
     
     GLint iTexSize;

     //before we proceed check the maximum texture size, we need 512x512
     glGetIntegerv(GL_MAX_TEXTURE_SIZE, &iTexSize);
     
     if (iTexSize<512) {
                       
         ::MessageBox(NULL, "Required texture size of 512 is not supported by your hardware", "Error", MB_OK);
         return;               
         }
     
     oWorld->lTextures = new GLuint[100];
     
     //Generate 100 texture name
     glGenTextures(100, oWorld->lTextures); 

     //set the method for storing pixels
	 glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

     //set the texture mode to modulate: 
	 glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	 glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
     
     //load the textures
     r_LoadTexture(t_cards + 0, "cards\\hearts_2.raw", true);
     r_LoadTexture(t_cards + 1, "cards\\hearts_3.raw", true);
     r_LoadTexture(t_cards + 2, "cards\\hearts_4.raw", true);
     r_LoadTexture(t_cards + 3, "cards\\hearts_5.raw", true);
     r_LoadTexture(t_cards + 4, "cards\\hearts_6.raw", true);
     r_LoadTexture(t_cards + 5, "cards\\hearts_7.raw", true);
     r_LoadTexture(t_cards + 6, "cards\\hearts_8.raw", true);
     r_LoadTexture(t_cards + 7, "cards\\hearts_9.raw", true);
     r_LoadTexture(t_cards + 8, "cards\\hearts_10.raw", true);
     r_LoadTexture(t_cards + 9, "cards\\hearts_J.raw", true);
     r_LoadTexture(t_cards + 10, "cards\\hearts_Q.raw", true);
     r_LoadTexture(t_cards + 11, "cards\\hearts_K.raw", true);
     r_LoadTexture(t_cards + 12, "cards\\hearts_A.raw", true);

     r_LoadTexture(t_cards + 13, "cards\\diamond_2.raw", true);
     r_LoadTexture(t_cards + 14, "cards\\diamond_3.raw", true);
     r_LoadTexture(t_cards + 15, "cards\\diamond_4.raw", true);
     r_LoadTexture(t_cards + 16, "cards\\diamond_5.raw", true);
     r_LoadTexture(t_cards + 17, "cards\\diamond_6.raw", true);
     r_LoadTexture(t_cards + 18, "cards\\diamond_7.raw", true);
     r_LoadTexture(t_cards + 19, "cards\\diamond_8.raw", true);
     r_LoadTexture(t_cards + 20, "cards\\diamond_9.raw", true);
     r_LoadTexture(t_cards + 21, "cards\\diamond_10.raw", true);
     r_LoadTexture(t_cards + 22, "cards\\diamond_J.raw", true);
     r_LoadTexture(t_cards + 23, "cards\\diamond_Q.raw", true);
     r_LoadTexture(t_cards + 24, "cards\\diamond_K.raw", true);
     r_LoadTexture(t_cards + 25, "cards\\diamond_A.raw", true);

     r_LoadTexture(t_cards + 26, "cards\\spades_2.raw", true);
     r_LoadTexture(t_cards + 27, "cards\\spades_3.raw", true);
     r_LoadTexture(t_cards + 28, "cards\\spades_4.raw", true);
     r_LoadTexture(t_cards + 29, "cards\\spades_5.raw", true);
     r_LoadTexture(t_cards + 30, "cards\\spades_6.raw", true);
     r_LoadTexture(t_cards + 31, "cards\\spades_7.raw", true);
     r_LoadTexture(t_cards + 32, "cards\\spades_8.raw", true);
     r_LoadTexture(t_cards + 33, "cards\\spades_9.raw", true);
     r_LoadTexture(t_cards + 34, "cards\\spades_10.raw", true);
     r_LoadTexture(t_cards + 35, "cards\\spades_J.raw", true);
     r_LoadTexture(t_cards + 36, "cards\\spades_Q.raw", true);
     r_LoadTexture(t_cards + 37, "cards\\spades_K.raw", true);
     r_LoadTexture(t_cards + 38, "cards\\spades_A.raw", true);

     r_LoadTexture(t_cards + 39, "cards\\clubs_2.raw", true);
     r_LoadTexture(t_cards + 40, "cards\\clubs_3.raw", true);
     r_LoadTexture(t_cards + 41, "cards\\clubs_4.raw", true);
     r_LoadTexture(t_cards + 42, "cards\\clubs_5.raw", true);
     r_LoadTexture(t_cards + 43, "cards\\clubs_6.raw", true);
     r_LoadTexture(t_cards + 44, "cards\\clubs_7.raw", true);
     r_LoadTexture(t_cards + 45, "cards\\clubs_8.raw", true);
     r_LoadTexture(t_cards + 46, "cards\\clubs_9.raw", true);
     r_LoadTexture(t_cards + 47, "cards\\clubs_10.raw", true);
     r_LoadTexture(t_cards + 48, "cards\\clubs_J.raw", true);
     r_LoadTexture(t_cards + 49, "cards\\clubs_Q.raw", true);
     r_LoadTexture(t_cards + 50, "cards\\clubs_K.raw", true);
     r_LoadTexture(t_cards + 51, "cards\\clubs_A.raw", true);

     r_LoadTexture(t_cards + 52, "cards\\joker_1.raw", true);
     r_LoadTexture(t_cards + 53, "cards\\joker_2.raw", true);

     r_LoadTexture(t_shim, "cards\\shim.raw", false);
     r_LoadTextureFont(t_font, "cards\\font_trebuchet.raw", "cards\\trebuchet_alpha.raw", "cards\\trebuchet.txt");
     r_LoadTexture(t_backsides + 0, "cards\\back_1.raw", true);
     r_LoadTexture(t_backsides + 1, "cards\\back_2.raw", true);
   
     r_LoadTexture(t_materials + 0, "cards\\wood.raw", true);
     r_LoadTexture(t_materials + 1, "cards\\vloer1.raw", true);
     r_LoadTexture(t_materials + 2, "cards\\vloer2.raw", true);
     r_LoadTexture(t_materials + 3, "cards\\muur1.raw", true);
     r_LoadTexture(t_materials + 4, "cards\\muur2.raw", true);
     r_LoadTextureAlpha2(t_materials + 5, "cards\\beam2.raw", "cards\\beam2_alpha.raw", true);
     r_LoadTexture(t_materials + 6, "cards\\distortion.raw", true);
     
     oMesh->r_LoadMeshX("monkey.x");
     //oRoom->r_LoadMeshX("room2.x");

	 lCurrentFont = 0;
     }

void cRenderer::r_LoadTextureFont(long lTextureIndex, char* pcFilename, 
								  char* pcFilenameAlpha, char* pcFilenameWidths) {

	sFont oFont;
	std::ifstream oInFile;
    std::string sLine;
	int iWidth;
	
	oFont.lTextureId = lTextureIndex;
     
	r_LoadTextureAlpha2(lTextureIndex, pcFilename, pcFilenameAlpha, true);

     //read the file data
	
	oInFile.open(pcFilenameWidths, std::ios_base::in);

	for (int iIndex=0; iIndex<256; iIndex++) {
     
		oInFile >> iWidth;
		oInFile.ignore(1,'\n'); 

		oFont.iCharacterWidth[iIndex] = iWidth;	
		}

	oFonts.push_back(oFont);
	}

void cRenderer::r_LoadTexture(long lTextureIndex, char* pcFilename, bool bMipMap) {
     
     BYTE* bTextureData;
     FILE* hFile;
     long lWidth;
     long lHeight;
     long lSize;
        
     //read the file data
     hFile = fopen(pcFilename, "rb");
     
     //determine the size of the bitmap file, files are raw format by default (RGB)
     fseek(hFile, 0, SEEK_END);
     lSize = ftell(hFile);
     fseek(hFile, 0, SEEK_SET);

     //determine the width and height base on the filesize
     lWidth = lHeight = (long)sqrt((float)lSize / 3);

     //allocate the memory required      
     bTextureData = (BYTE*)malloc(lSize);
     
     if (bTextureData==NULL) {
                             
         ::MessageBox(NULL, "Out of memory while loading textures", "Error", MB_OK);
         return;
         }
        
     fread(bTextureData, 3, lWidth*lHeight, hFile);
        
     fclose(hFile); 
       
    //bind the texture
    
    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[lTextureIndex]);
    
    if (bMipMap) {
    
      //Construct multiple textures of the give texture
       
       gluBuild2DMipmaps(GL_TEXTURE_2D,     // texture to specify
                 GL_RGB,                    // internal texture storage format
                 lWidth,                    // texture width
                 lHeight,                   // texture height
                 GL_RGB,                    // pixel format
                 GL_UNSIGNED_BYTE,	        // color component format
                 bTextureData);                 // pointer to texture image*/
    
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
       }
    else {
    
        //Set the texture as a single texture
    
       glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, lWidth, lHeight, 0, GL_RGB, GL_UNSIGNED_BYTE, bTextureData); 
    
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    }
    
    free(bTextureData);    
}


void cRenderer::r_LoadTextureAlpha(long lTextureIndex, char* pcFilename, bool bMipMap) {
     
     BYTE* bTextureData;
     BYTE* bAlphaData;
     FILE* hFile;
     long lWidth;
     long lHeight;
     long lSize;
        
     //read the file data
     hFile = fopen(pcFilename, "rb");
     
     //determine the size of the bitmap file, files are raw format by default (RGB)
     fseek(hFile, 0, SEEK_END);
     lSize = ftell(hFile);
     fseek(hFile, 0, SEEK_SET);

     //determine the width and height base on the filesize
     lWidth = lHeight = (long)sqrt((float)lSize / 3);

     //allocate the memory required      
     bTextureData = (BYTE*)malloc(lSize);
     
     if (bTextureData==NULL) {
                             
         ::MessageBox(NULL, "Out of memory while loading textures", "Error", MB_OK);
         return;
         }
        
     fread(bTextureData, 3, lWidth*lHeight, hFile);
        
     fclose(hFile); 
       
    //bind the texture
    
    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[lTextureIndex]);
    
        //Set the texture as a single texture
        bAlphaData = (BYTE*)malloc(lWidth * lHeight * 4);

        for (long lX = 0; lX < lWidth; lX++) {
            
            for (long lY = 0; lY < lHeight; lY++) {
                
                bAlphaData[lY * lWidth * 4 + lX * 4] = bTextureData[lY * lWidth * 3 + lX * 3];
                bAlphaData[lY * lWidth * 4 + lX * 4 + 1] = bTextureData[lY * lWidth * 3 + lX * 3 + 1];
                bAlphaData[lY * lWidth * 4 + lX * 4 + 2] = bTextureData[lY * lWidth * 3 + lX * 3 + 2];
				bAlphaData[lY * lWidth * 4 + lX * 4 + 3] = max(bTextureData[lY * lWidth * 3 + lX * 3], max(bTextureData[lY * lWidth * 3 + lX * 3 + 1], bTextureData[lY * lWidth * 3 + lX * 3 + 2]))==0 ? 0 : 255;
                
                //if (bAlphaData[lY * lWidth * 4 + lX * 4 + 3] > 0) bAlphaData[lY * lWidth * 4 + lX * 4 + 3] = 255;
                }
            }

    if (bMipMap) {
    
      //Construct multiple textures of the give texture
       
       gluBuild2DMipmaps(GL_TEXTURE_2D,     // texture to specify
                 GL_RGBA,                    // internal texture storage format
                 lWidth,                    // texture width
                 lHeight,                   // texture height
                 GL_RGBA,                    // pixel format
                 GL_UNSIGNED_BYTE,	        // color component format
                 bAlphaData);                 // pointer to texture image*/
    
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
       }
    else {
    
    
    
       glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, lWidth, lHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, bAlphaData); 
    
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
       
    }
    
       free(bAlphaData);
    free(bTextureData);    
}

void cRenderer::r_LoadTextureAlpha2(long lTextureIndex, char* pcFilename, char* pcFilenameAlpha, bool bMipMap) {
     
     BYTE* bTextureData;
     BYTE* bAlphaData;
     BYTE* bImageData;
     FILE* hFile;
     long lWidth;
     long lHeight;
     long lSize;
        
     //read the file data
     hFile = fopen(pcFilename, "rb");
     
     //determine the size of the bitmap file, files are raw format by default (RGB)
     fseek(hFile, 0, SEEK_END);
     lSize = ftell(hFile);
     fseek(hFile, 0, SEEK_SET);

     //determine the width and height base on the filesize
     lWidth = lHeight = (long)sqrt((float)lSize / 3);

     //allocate the memory required      
     bImageData = (BYTE*)malloc(lSize);
     bAlphaData = (BYTE*)malloc(lWidth * lHeight);
     
     if (bAlphaData==NULL || bImageData==NULL) {
                             
         ::MessageBox(NULL, "Out of memory while loading textures", "Error", MB_OK);
         return;
         }
        
     fread(bImageData, 3, lWidth*lHeight, hFile);
        
     fclose(hFile); 
       
     hFile = fopen(pcFilenameAlpha, "rb");

     fread(bAlphaData, 1, lWidth*lHeight, hFile);
        
     fclose(hFile); 


    //bind the texture
    
    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[lTextureIndex]);
    
        //Set the texture as a single texture
        bTextureData = (BYTE*)malloc(lWidth * lHeight * 4);

        for (long lX = 0; lX < lWidth; lX++) {
            
            for (long lY = 0; lY < lHeight; lY++) {
                
                bTextureData[lY * lWidth * 4 + lX * 4] = bImageData[lY * lWidth * 3 + lX * 3];
                bTextureData[lY * lWidth * 4 + lX * 4 + 1] = bImageData[lY * lWidth * 3 + lX * 3 + 1];
                bTextureData[lY * lWidth * 4 + lX * 4 + 2] = bImageData[lY * lWidth * 3 + lX * 3 + 2];
                bTextureData[lY * lWidth * 4 + lX * 4 + 3] = bAlphaData[lY * lWidth + lX];
                
                //if (bAlphaData[lY * lWidth * 4 + lX * 4 + 3] > 0) bAlphaData[lY * lWidth * 4 + lX * 4 + 3] = 255;
                }
            }

    if (bMipMap) {
    
      //Construct multiple textures of the give texture
       
       gluBuild2DMipmaps(GL_TEXTURE_2D,     // texture to specify
                 GL_RGBA,                    // internal texture storage format
                 lWidth,                    // texture width
                 lHeight,                   // texture height
                 GL_RGBA,                    // pixel format
                 GL_UNSIGNED_BYTE,	        // color component format
                 bTextureData);                 // pointer to texture image*/
    
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
       glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
       }
    else {
    
    
    
       glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, lWidth, lHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, bTextureData); 
    
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
       glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
       
    }
    
       free(bAlphaData);
    free(bTextureData);
    free(bImageData);    
}

void cRenderer::r_DrawCards(sDeck &poDeck, GLenum mode) {
     
	sCard* oCard;
	
	for (long lIndex=0; lIndex < poDeck.oCards.size(); lIndex++) {
         
		oCard = poDeck.oCards[lIndex]; 
		
		if (oCard->bVisible) {
            
			glPushMatrix();

			if (mode == GL_SELECT) {
				glLoadName(lIndex);
                }
            
			r_DrawCard(oCard, mode, true);
 
			glPopMatrix();
			}
		}
     }


void cRenderer::r_DrawPlayerHand(sPlayer &poPlayers, sDeck &poDeck, GLenum mode, bool pbDrawFront=true) {
    
    //make the changes to the matrix
    glPushMatrix();
    
    //glTranslatef(poPlayers.vPosition[0], poPlayers.vPosition[1], poPlayers.vPosition[2]);
    //glRotatef(poPlayers.vRotation[1], 0, 1, 0); 
   // glRotatef(poPlayers.vRotation[0], 1, 0, 0);
    
	//position the cards in the hand
   // r_PositionCardsInHand(poPlayers, oWorld->oDeck);
    
    //do the drawing
	for (long lIndex=0; lIndex<(long)poPlayers.oCards.size(); lIndex++) {
        
		if (mode == GL_SELECT) {
			glLoadName(poPlayers.oCards[lIndex]->lId);
			}
		r_DrawCard(poPlayers.oCards[lIndex], mode, pbDrawFront);
		} 
     
	//restore the matrix
    glPopMatrix();
    }


void cRenderer::r_DrawCard(sCard *poCard, GLenum mode, bool pbDrawFront = true) {

    bool bShaderInUse;
	
	if (poCard->bFake) return;
	
	glPushMatrix();

    //position the card in the world
	glTranslatef(poCard->vPosition[0], poCard->vPosition[1], poCard->vPosition[2]);
	
	//draw the lines to check the positioning of the card
	if (oWorld->RenderOptions.bDrawCardLines) { 
		glBegin(GL_LINES);
		   glVertex3f(0.0f, 0.0f, -0.5f);
		   glVertex3f(0.0f, 0.0f, 0.5f);
		glEnd();
	}
   
	//Rotate the card and if set draw the card
	glRotatef(poCard->vRotation[0] / pi * 180.0f, 0, 1, 0);

	if (oWorld->RenderOptions.bDrawCardLines) { 
		glBegin(GL_LINES);
		   glVertex3f(-0.5f, 0.0f, 0.0f);
		   glVertex3f(0.5f, 0.0f, 0.0f);
		glEnd();
	}
	glRotatef(poCard->vRotation[1] / pi * 180.0f, 1, 0, 0);

	if (oWorld->RenderOptions.bDrawCardLines) { 
		glBegin(GL_LINES);
		   glVertex3f(0.0f, 0.0f, 0.0f);
		   glVertex3f(0.0f, poCard->fTranslation +0.25f, 0.0f);
		glEnd();
	}
    glRotatef(poCard->vRotation[2] / pi * 180.0f, 0, 1, 0);
    glTranslatef(0, poCard->fTranslation, 0);

    glRotatef(poCard->vRotation[3] / pi * 180.0f, 1, 0, 0);


	//glRotatef(lIncrement,1,1,0);
    glScalef(poCard->fScale * 0.33f, poCard->fScale*card_thickness/2, poCard->fScale*0.5f);

	//set the transparency of the card
	if (oWorld->oDeck.fTransparancy * poCard->fTransparancy < 1.0f) {
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		}

	GLfloat mat_unselected[] = { 0.5f, 0.5f, 0.5f, oWorld->oDeck.fTransparancy * poCard->fTransparancy};
	GLfloat mat_selected[] = { 0.8f, 0.8f, 0.8f, oWorld->oDeck.fTransparancy * poCard->fTransparancy };
	
	if (!poCard->bSelected) {

		//set the material attributes for a non selected card
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,   mat_unselected);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   mat_unselected);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  oWorld->MaterialConstants.no_mat);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, oWorld->MaterialConstants.no_mat);
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION,  oWorld->MaterialConstants.no_mat);
		}
	else {
       
		//set the material attributes for a selected card
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,   mat_selected);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   mat_selected);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  oWorld->MaterialConstants.no_mat);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, oWorld->MaterialConstants.no_mat);
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION,  oWorld->MaterialConstants.no_mat);
		}

	//Draw the card its cube

    //set the texture for the front side and draw the front side
	if ((!poCard->bSecret && pbDrawFront) || oWorld->bDebug) {
		glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[poCard->lTexture]);
		
		int maxSupportedAnisotrophy = 1;
		glGetIntegerv( GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &maxSupportedAnisotrophy );
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, oWorld->RenderOptions.bAnisotropicFiltering ? min(4.0f, maxSupportedAnisotrophy) : 1.0f);
		}
	else {
		glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_shim]);
		}

	glNormal3f(0, 1, 0);
    
	if (poCard->bSelected) {
		
		bShaderInUse = r_UseShader(s_cardselected);
	
		if (bShaderInUse) {
		
			GLint uniform_location;
			uniform_location = glGetUniformLocationARB(oShaders[1].hProgramObject, "fTime");
			float fTime;

			fTime = (float)(oSystem->GetTickCount()%10000)/10000.0f;
			glUniform1fARB(uniform_location, fTime);

			uniform_location = glGetUniformLocationARB(oShaders[1].hProgramObject, "noise");
			glUniform1iARB(uniform_location, 1);

			/*glActiveTextureARB(GL_TEXTURE0_ARB);
			glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_cards + poCard->iValue + poCard->iColor*13]);	
			glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials+6]);	
			glEnable(GL_TEXTURE_2D);*/

			glActiveTextureARB(GL_TEXTURE1_ARB);
			glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials+6]);	
			glEnable(GL_TEXTURE_2D);
			}
		}
	
	glMatrixMode(GL_TEXTURE);
	glPushMatrix();

	if (poCard->bFlipTexture) {

		glRotatef(180, 0, 0, 1);
		}
		
		
	glBegin(GL_POLYGON);
		//curve bottom left 
		glTexCoord2f(0.05f, 0.0f);  glVertex3f(-0.9f,  1.0f, -1.0f);

		glTexCoord2f(0.03f, 0.01f); glVertex3f(-0.95f, 1.0f, -0.98f);
		glTexCoord2f(0.01f, 0.03f); glVertex3f(-0.98f, 1.0f, -0.95f);

		glTexCoord2f(0.0f, 0.05f);  glVertex3f(-1.0f,  1.0f, -0.9f);


		//curve top left 
		glTexCoord2f(0.0f, 0.95f);  glVertex3f(-1.0f,  1.0f, 0.9f); 

		glTexCoord2f(0.01f, 0.97f); glVertex3f(-0.98f, 1.0f, 0.95f);
		glTexCoord2f(0.03f, 0.98f); glVertex3f(-0.95f, 1.0f, 0.98f);
    	
		glTexCoord2f(0.05f, 1.0f);  glVertex3f(-0.9f,  1.0f, 1.0f);


		//curve top right
		glTexCoord2f(0.95f, 1.0f);  glVertex3f( 0.9f, 1.0f, 1.0f);

		glTexCoord2f(0.97f, 0.99f); glVertex3f( 0.95f, 1.0f,  0.98f); 
		glTexCoord2f(0.99f, 0.97f); glVertex3f( 0.98f, 1.0f, 0.95f);
    	
		glTexCoord2f(1.0f, 0.95f);  glVertex3f( 1.0f, 1.0f,  0.9f);


		//curve bottom right
		glTexCoord2f(1.0f, 0.05f);  glVertex3f( 1.0f, 1.0f,  -0.9f); 

		glTexCoord2f(0.99f, 0.03f); glVertex3f( 0.98f, 1.0f, -0.95f);
		glTexCoord2f(0.97f, 0.01f); glVertex3f( 0.95f, 1.0f, -0.98f);

		glTexCoord2f(0.95f, 0.0f);  glVertex3f( 0.9f, 1.0f,  -1.0f); 

	glEnd();

	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	
    //set the texture for the back side and r_Draw the back side
	if (bShaderInUse) {
	
		glActiveTextureARB(GL_TEXTURE1_ARB);
		glDisable(GL_TEXTURE_2D);
		glActiveTextureARB(GL_TEXTURE0_ARB);
		}

	glActiveTextureARB(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[poCard->iBackside]);

     if (oWorld->lCardDisplayList !=0 && !poCard->bSelected) {
        glCallList(oWorld->lCardDisplayList);                            
        }
     else {

        if (!poCard->bSelected) {
			oWorld->lCardDisplayList = glGenLists(1);
     
			glNewList(oWorld->lCardDisplayList, GL_COMPILE);
			}

        glNormal3f(0, -1, 0);
    
    	glBegin(GL_POLYGON);
        	//curve bottom left 
            glTexCoord2f(0.0f, 0.05f); glVertex3f( -1.0f,  -1.0f, -0.9f);
    
        	glTexCoord2f(0.01f, 0.03f); glVertex3f( -0.98f, -1.0f, -0.95f);
        	glTexCoord2f(0.03f, 0.01f); glVertex3f( -0.95f, -1.0f, -0.98f);
    
        	glTexCoord2f(0.05f, 0.0f); glVertex3f( -0.9f,  -1.0f, -1.0f);
    
    
            //curve bottom right 
        	glTexCoord2f(0.95f, 0.0f); glVertex3f(  0.9f,  -1.0f, -1.0f); 
    
        	glTexCoord2f(0.97f, 0.01f); glVertex3f(  0.95f, -1.0f, -0.98f);
        	glTexCoord2f(0.99f, 0.03f); glVertex3f(  0.98f, -1.0f, -0.95f);
        	
            glTexCoord2f(1.0f, 0.05f); glVertex3f(  1.0f,  -1.0f, -0.9f);
    
    
            //curve top right
            glTexCoord2f(1.0f, 0.95f); glVertex3f(  1.0f, -1.0f, 0.9f);
    
        	glTexCoord2f(0.99f, 0.97f); glVertex3f(  0.98f, -1.0f,  0.95f); 
        	glTexCoord2f(0.97f, 0.99f); glVertex3f(  0.95f, -1.0f, 0.98f);
        	
            glTexCoord2f(0.95f, 1.0f); glVertex3f(  0.9f, -1.0f,  1.0f);
    
    
            //curve top left
        	glTexCoord2f(0.05f, 1.0f); glVertex3f( -0.9f, -1.0f,  1.0f); 
    
        	glTexCoord2f(0.03f, 0.99f); glVertex3f( -0.95f, -1.0f, 0.98f);
        	glTexCoord2f(0.01f, 0.97f); glVertex3f( -0.98f, -1.0f, 0.95f);
    
        	glTexCoord2f(0.0f, 0.95f); glVertex3f( -1.0f, -1.0f,  0.9f); 
    
    	glEnd();
    
    
        //set the texture for the side and r_Draw the sides
        
        glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_shim]);
    	glColor3f( 1.0f, 1.0f, 1.0f ); 
        glBegin(GL_QUAD_STRIP);
    
    
            glNormal3f(-1, 0, 0);
    
            glTexCoord2f(0, 0); glVertex3f( -1.0f,  -1.0f, -0.9f);
            glTexCoord2f(0, 1); glVertex3f( -1.0f,  1.0f, -0.9f);
    
        	glTexCoord2f(1, 1); glVertex3f( -0.98f, -1.0f, -0.95f);
        	glTexCoord2f(1, 0); glVertex3f( -0.98f, 1.0f, -0.95f);
    
        	glTexCoord2f(0, 0); glVertex3f( -0.95f, -1.0f, -0.98f);
        	glTexCoord2f(0, 1); glVertex3f( -0.95f, 1.0f, -0.98f);
    
        	glTexCoord2f(1, 1); glVertex3f( -0.9f,  -1.0f, -1.0f);
            glNormal3f(0, 0, -1);
        	glTexCoord2f(0, 1); glVertex3f( -0.9f,  1.0f, -1.0f);
    
    
    
            //curve bottom right 
        	glTexCoord2f(0, 0); glVertex3f(  0.9f,  -1.0f, -1.0f); 
        	glTexCoord2f(0, 1); glVertex3f(  0.9f,  1.0f, -1.0f); 
    
        	glTexCoord2f(1, 1); glVertex3f(  0.95f, -1.0f, -0.98f);
        	glTexCoord2f(1, 0); glVertex3f(  0.95f, 1.0f, -0.98f);
    
        	glTexCoord2f(0, 0); glVertex3f(  0.98f, -1.0f, -0.95f);
        	glTexCoord2f(0, 1); glVertex3f(  0.98f, 1.0f, -0.95f);
        	
            glTexCoord2f(1, 1); glVertex3f(  1.0f,  -1.0f, -0.9f);
            glNormal3f(1, 0, 0);
            glTexCoord2f(1, 0); glVertex3f(  1.0f,  1.0f, -0.9f);
    
    
            //curve top right
            glTexCoord2f(0, 0); glVertex3f(  1.0f, -1.0f, 0.9f);
            glTexCoord2f(0, 1); glVertex3f(  1.0f, 1.0f, 0.9f);
    
        	glTexCoord2f(1, 1); glVertex3f(  0.98f, -1.0f,  0.95f); 
        	glTexCoord2f(1, 0); glVertex3f(  0.98f, 1.0f,  0.95f); 
    
        	glTexCoord2f(0, 0); glVertex3f(  0.95f, -1.0f, 0.98f);
        	glTexCoord2f(0, 1); glVertex3f(  0.95f, 1.0f, 0.98f);
        	
            glTexCoord2f(1, 1); glVertex3f(  0.9f, -1.0f,  1.0f);
            glNormal3f(0, 0, 1);
            glTexCoord2f(1, 0); glVertex3f(  0.9f, 1.0f,  1.0f);
    
    
            //curve top left
        	glTexCoord2f(0, 0); glVertex3f( -0.9f, -1.0f,  1.0f); 
        	glTexCoord2f(0, 1); glVertex3f( -0.9f, 1.0f,  1.0f); 
    
        	glTexCoord2f(1, 1); glVertex3f( -0.95f, -1.0f, 0.98f);
        	glTexCoord2f(1, 0); glVertex3f( -0.95f, 1.0f, 0.98f);
    
        	glTexCoord2f(0, 0); glVertex3f( -0.98f, -1.0f, 0.95f);
        	glTexCoord2f(0, 1); glVertex3f( -0.98f, 1.0f, 0.95f);
    
        	glTexCoord2f(0.0f, 0.95f); glVertex3f( -1.0f, -1.0f,  0.9f); 
        	glTexCoord2f(0.0f, 0.95f); glVertex3f( -1.0f, 1.0f,  0.9f); 
    
            glTexCoord2f(1, 1); glVertex3f( -1.0f,  -1.0f, -0.9f);
            glNormal3f(-1, 0, 0);
            glTexCoord2f(1, 0); glVertex3f( -1.0f,  1.0f, -0.9f);
    

    	glEnd();

	if (!poCard->bSelected) glEndList();

	if (poCard->bSelected) {
                               
		if (bShaderInUse) {

			r_TurnOffShader();
			}
		else {
		 
			GLfloat mat_selected[] = { 1.3f, 1.3f, 1.3f, 1.0f };

			glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,   mat_selected);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   mat_selected);
			glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  mat_selected);
			glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, oWorld->MaterialConstants.no_mat);
			glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION,  mat_selected);

			glEnable(GL_BLEND);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

			glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials+5]);
	                               
			glBegin(GL_QUADS);
	    
				glTexCoord2f(0, 0); glVertex3f( -1.06f,  1.05f, 1.06f);
				glTexCoord2f(1, 0); glVertex3f( 1.06f,  1.05f, 1.06f);
				glTexCoord2f(1, 1); glVertex3f( 1.06f,  1.05f, -1.06f);
				glTexCoord2f(0, 1); glVertex3f( -1.06f,  1.05f, -1.06f);
	        
			glEnd(); 
	    
			glDisable(GL_BLEND);
			}
        }
    }
     
	glPopMatrix();
    }

void cRenderer::r_DrawSquare()
{
	glColor3f( 1.0f, 1.0f, 1.0f ); 
    glBegin(GL_QUADS);
	glVertex3f(-0.2f,0,-0.2f);
	glVertex3f(-0.2f,0,0.2f);
	glVertex3f(+0.2f,0,0.2f);
	glVertex3f(+0.2f,0,-0.2f);
	glEnd();
}



void cRenderer::r_DrawHud(GLenum mode) {

	r_DrawWindows(mode);

return;
     GLfloat red[4] = { 0.2, 0.2, 0.2, 0.7 };
    glColor4fv(red);
    glDisable(GL_TEXTURE_2D); 
     
     glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA_SATURATE, GL_ONE_MINUS_SRC_ALPHA);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE);
     
     glBegin(GL_POLYGON);
     
         glVertex2f(2.0f, oSystem->iWindowHeight - 2.0f);
         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight - 2.0f);
         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);
         glVertex2f(2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);
     
     glEnd();
     
     glDisable(GL_BLEND);

    GLfloat white[4] = { 1.0, 1.0, 1.0, 1.0 };
    glColor4fv(white);

     /*glBegin(GL_LINES);
     
         glVertex2f(2.0f, oSystem->iWindowHeight - 2.0f);
         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight - 2.0f);

         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight - 2.0f);
         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);

         glVertex2f(oSystem->iWindowWidth - 2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);
         glVertex2f(2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);
     
         glVertex2f(2.0f, oSystem->iWindowHeight * 0.8f - 2.0f);
         glVertex2f(2.0f, oSystem->iWindowHeight - 2.0f);

     glEnd();*/
     
         glEnable(GL_TEXTURE_2D); 

     }

void cRenderer::r_DrawWindows(GLenum mode) {

	GLfloat vBackgroundColor[4];
	GLfloat vBorderColor[4];
	sRect	oRect;
	sWindow oWindow;
	sText	oText;
	sButton oButton;

	bool bBlend;
	bool bTexture2D;
	bool bLighting;
	
	//save OpenGL settings/states
	bBlend =	 glIsEnabled(GL_BLEND);
	bTexture2D = glIsEnabled(GL_TEXTURE_2D);
	bLighting =	 glIsEnabled(GL_LIGHTING);

	//change OpenGl settings/states
	glDisable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D); 
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	//glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_font]);

	for (long lIndex=0; lIndex<oWorld->oWindows.size(); lIndex++) {

		oWindow = oWorld->oWindows[lIndex];

		vBorderColor[0] = 0.5f;
		vBorderColor[1] = 0.5f;
		vBorderColor[2] = 0.5f;
		vBorderColor[3] = oWindow.fTransparancy;

		vBackgroundColor[0] = oWindow.vBackgroundColor[0];
		vBackgroundColor[1] = oWindow.vBackgroundColor[1];
		vBackgroundColor[2] = oWindow.vBackgroundColor[2];
		vBackgroundColor[3] = oWindow.fTransparancy;
		
		/*glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  vBackgroundColor);*/
		glColor4fv(vBackgroundColor);


		float fScaleX, fScaleY;

		fScaleX = (oSystem->iGameWindowWidth > oSystem->iGameWindowHeight) ? (float)oSystem->iGameWindowHeight / (float)oSystem->iGameWindowWidth : 1.0f;
		fScaleY = (oSystem->iGameWindowWidth > oSystem->iGameWindowHeight) ? 1.0f : (float)oSystem->iGameWindowWidth / (float)oSystem->iGameWindowHeight;

		fScaleX = 1.0f;
		fScaleY = 1.0f;

		//Set the window size
		oRect.iLeft = 0;
		oRect.iTop = 0;

		switch (oWindow.eHorizontalSize) {

			case sizePercentage:
				oRect.iRight = oSystem->iGameWindowWidth * oWindow.vSize[0];
				break;

			default:
				oRect.iRight = oWindow.vSize[0];
			}

		switch (oWindow.eVerticalSize) {

			case sizePercentage:
				oRect.iBottom = oSystem->iGameWindowHeight * oWindow.vSize[1];
				break;

			default:
				oRect.iBottom = oWindow.vSize[1];
			}

		//Position the window in the screen
		switch (oWindow.eHorizontalPosition) {

			case posLeft:
				oRect.iLeft = oWindow.vPosition[0];
				break;

			case posRight:
				oRect.iLeft = oSystem->iGameWindowWidth - oRect.iRight - oWindow.vPosition[0];
				break;

			case posPercentage:
				oRect.iLeft = oSystem->iGameWindowWidth * oWindow.vPosition[0];
				break;

			case posCenter:
				oRect.iLeft = (oSystem->iGameWindowWidth - oRect.iRight) / 2;
				break;

			default:
				oRect.iLeft = oWindow.vPosition[0];
			}

		oRect.iRight += oRect.iLeft;


		switch (oWindow.eVerticalPosition) {

			case posTop:
				oRect.iTop = oWindow.vPosition[1];
				break;

			case posBottom:
				oRect.iTop = oSystem->iGameWindowHeight - oRect.iBottom - oWindow.vPosition[1];
				break;

			case posPercentage:
				oRect.iTop = oSystem->iGameWindowHeight * oWindow.vPosition[1];
				break;

			case posCenter:
				oRect.iTop = (oSystem->iGameWindowHeight - oRect.iBottom) / 2;
				break;

			default:
				oRect.iTop = oWindow.vPosition[1];
			}

		oRect.iBottom += oRect.iTop;

		//Draw the window
		if (mode == GL_SELECT) {
			glLoadName(200 + oWindow.lWindowId * 10);
			}
		
		r_Draw2DShape(shapeRoundedRectangle, oRect, true);

		glColor4fv(vBorderColor);
		glLineWidth(0.1f);

		r_Draw2DShape(shapeRoundedRectangle, oRect, false);

		for (long lIndex=0; lIndex<oWindow.oTexts.size(); lIndex++) {

			oText = oWindow.oTexts[lIndex];
			
			oText.iX += oRect.iLeft;
			oText.iY += oRect.iTop;

			oText.cTextRect.iLeft += oRect.iLeft;
			oText.cTextRect.iRight += oRect.iLeft;
			oText.cTextRect.iTop += oRect.iTop;
			oText.cTextRect.iBottom += oRect.iTop;

			
			//draw a rectangle to show the text rectangle for debugging purposes

			/*glBegin(GL_POLYGON);
	     
				glVertex2f(oText.cTextRect.iLeft, oText.cTextRect.iTop);
				glVertex2f(oText.cTextRect.iRight, oText.cTextRect.iTop);
				glVertex2f(oText.cTextRect.iRight, oText.cTextRect.iBottom);
				glVertex2f(oText.cTextRect.iLeft, oText.cTextRect.iBottom);
	     
			glEnd();*/

			r_DrawText(oText);
			}

		//Draw the buttons
		for (long lIndex=0; lIndex<oWindow.oButtons.size(); lIndex++) {

			oButton = oWindow.oButtons[lIndex];
			
			oButton.cButtonRect.iLeft += oRect.iLeft;
			oButton.cButtonRect.iRight += oRect.iLeft;
			oButton.cButtonRect.iTop += oRect.iTop;
			oButton.cButtonRect.iBottom += oRect.iTop;

			oButton.fTransparancy *= oWindow.fTransparancy;

			if (mode == GL_SELECT) {
				glLoadName(200 + oWindow.lWindowId * 10 + lIndex + 1);
				}

			r_DrawButton(oButton);

			if (mode == GL_SELECT) {
				glLoadName(200);
				}
			}
		}
	
	//Reset OpenGL states
	if (!bBlend)		glDisable(GL_BLEND); else glEnable(GL_BLEND);
    if (!bTexture2D)	glDisable(GL_TEXTURE_2D); else glEnable(GL_TEXTURE_2D);
    if (!bLighting)		glDisable(GL_LIGHTING); else glEnable(GL_LIGHTING);
		
	vBackgroundColor[0] = 1.0f;
	vBackgroundColor[1] = 1.0f;
	vBackgroundColor[2] = 1.0f;
	vBackgroundColor[3] = 1.0f;
	glColor4fv(vBackgroundColor);
	}


//Draw a Shape in 2D (HUD layer), all settings (color, transparancy
//textures etc) has to be set by the caller function

void cRenderer::r_Draw2DShape(eShape peShape, sRect poRect, bool bFill) {

	float lRoundSize = (poRect.iRight - poRect.iLeft) * 0.03; //oSystem->iGameWindowWidth / 70;
	float x;
	float y;

	switch (peShape) {

		case shapeRoundedRectangle:

			if (bFill) {
				glBegin(GL_POLYGON);
	     
					for (long lAngle = 0; lAngle <= 90; lAngle+=5) {
						x = poRect.iLeft + lRoundSize * (1 - cos((lAngle * pi) / 180.0f));
						y = poRect.iTop  + lRoundSize * (1 - sin((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}
				
					//glTexCoord2f(0,0.2);
					//glVertex2f(poRect.iLeft + lRoundSize, poRect.iTop);
					
					for (long lAngle = 0; lAngle <= 90; lAngle+=5) {
						x = poRect.iRight - lRoundSize * (1 - sin((lAngle * pi) / 180.0f));
						y = poRect.iTop + lRoundSize * (1 - cos((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}
						
					for (long lAngle = 0; lAngle < 90; lAngle+=5) {
						x = poRect.iRight - lRoundSize * (1 - cos((lAngle * pi) / 180.0f));
						y = poRect.iBottom  - lRoundSize * (1 - sin((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}

					for (long lAngle = 0; lAngle < 90; lAngle+=5) {
						x = poRect.iLeft + lRoundSize * (1 - sin((lAngle * pi) / 180.0f));
						y = poRect.iBottom  - lRoundSize * (1 - cos((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}

				glEnd();
				}
			else {
			
				glBegin(GL_LINE_LOOP);

					for (long lAngle = 0; lAngle <= 90; lAngle+=5) {
						x = poRect.iLeft + lRoundSize * (1 - cos((lAngle * pi) / 180.0f));
						y = poRect.iTop  + lRoundSize * (1 - sin((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}
				
					for (long lAngle = 0; lAngle <= 90; lAngle+=5) {
						x = poRect.iRight - lRoundSize * (1 - sin((lAngle * pi) / 180.0f));
						y = poRect.iTop + lRoundSize * (1 - cos((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}
						
					for (long lAngle = 0; lAngle < 90; lAngle+=5) {
						x = poRect.iRight - lRoundSize * (1 - cos((lAngle * pi) / 180.0f));
						y = poRect.iBottom  - lRoundSize * (1 - sin((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}

					for (long lAngle = 0; lAngle < 90; lAngle+=5) {
						x = poRect.iLeft + lRoundSize * (1 - sin((lAngle * pi) / 180.0f));
						y = poRect.iBottom  - lRoundSize * (1 - cos((lAngle * pi) / 180.0f));

						glVertex2f(x, y);
						}

				glEnd();
				}
			break;

		case shapeRectangle:

			if (bFill) {
				glBegin(GL_POLYGON);
				
					glTexCoord2f(1, 1);
					glVertex2f(poRect.iRight, poRect.iTop);
					glTexCoord2f(1, 0);
					glVertex2f(poRect.iRight, poRect.iBottom);
					glTexCoord2f(0, 0);
					glVertex2f(poRect.iLeft, poRect.iBottom);
					glTexCoord2f(0, 1);
					glVertex2f(poRect.iLeft, poRect.iTop);

				glEnd();
				}
			else {

				glBegin(GL_LINE_LOOP);

					glVertex2f(poRect.iLeft, poRect.iTop);
					glVertex2f(poRect.iRight, poRect.iTop);
					glVertex2f(poRect.iRight, poRect.iBottom);
					glVertex2f(poRect.iLeft, poRect.iBottom);

				glEnd();
				}

			break;
		}
	}

		
//Draw a button on the screen

void cRenderer::r_DrawButton(sButton &poButton) {

	sText oCaption;
	
	poButton.vBackgroundColor[3] = poButton.fTransparancy;
	glColor4fv(poButton.vBackgroundColor);

	//poButton.cButtonRect.iTop = oSystem->iGameWindowHeight - poButton.cButtonRect.iTop;
	//poButton.cButtonRect.iBottom = oSystem->iGameWindowHeight - poButton.cButtonRect.iBottom;

	r_Draw2DShape(shapeRoundedRectangle, poButton.cButtonRect, true);

	oCaption.cTextRect = poButton.cButtonRect;
	oCaption.dStyle = styleAlignCenterH | styleAlignCenterV | styleSizeFixed;
	strcpy(oCaption.cText, poButton.cCaption);
	oCaption.vColor[0] = 1.0f;
	oCaption.vColor[1] = 1.0f;
	oCaption.vColor[2] = 1.0f;
	oCaption.vColor[3] = 1.0f;
	oCaption.fSize = wndButtonFontSize;
	oCaption.fTransparancy = poButton.fTransparancy;

	r_DrawText(oCaption);
	}



void cRenderer::r_DrawTrianglesRecursive(vec_f v1, vec_f v2, vec_f v3, long lDepth) {
     
     vec_f t1, t2, nf;
     
     if (lDepth>0) {
     
         t1[0] = (v2[0] + v3[0]) / 2;
         t1[1] = (v2[1] + v3[1]) / 2;
         t1[2] = (v2[2] + v3[2]) / 2;
         
         r_DrawTrianglesRecursive(t1, v1, v2, lDepth-1);
         r_DrawTrianglesRecursive(t1, v3, v1, lDepth-1);
         }
     else {
          
        t1[0] = v2[0] - v1[0];  
        t1[1] = v2[1] - v1[1];  
        t1[2] = v2[2] - v1[2];  

        t2[0] = v3[0] - v1[0];  
        t2[1] = v3[1] - v1[1];  
        t2[2] = v3[2] - v1[2]; 
        
        CrossProduct(t1, t2, nf);
        NormalizeVector(nf, nf);
        
        glNormal3f(nf[0], nf[1], nf[2]);
          
        glBegin(GL_TRIANGLES);
        	glTexCoord2f((v1[0] + 1.0f)/2 * oWorld->fTextureRepeat, (v1[1] + 1.0f)/2 * oWorld->fTextureRepeat); glVertex3f(v1[0], v1[1], v1[2]);
        	glTexCoord2f((v2[0] + 1.0f)/2 * oWorld->fTextureRepeat, (v2[1] + 1.0f)/2 * oWorld->fTextureRepeat); glVertex3f(v2[0], v2[1], v2[2]);
        	glTexCoord2f((v3[0] + 1.0f)/2 * oWorld->fTextureRepeat, (v3[1] + 1.0f)/2 * oWorld->fTextureRepeat); glVertex3f(v3[0], v3[1], v3[2]);
        glEnd();
          }
     }

void cRenderer::r_DrawQuadsRecursive(vec_f v1, vec_f v2, long lDepth) {
     
     vec_f t1;
     vec_f t2, t3;
     if (lDepth>0) {
        
        //divide the quad into four pieces
        t1[0] = (v1[0] + v2[0]) / 2;
        t1[1] = (v1[1] + v2[1]) / 2;
        t1[2] = (v1[2] + v2[2]) / 2;
        t1[1] += 0.1f;
        
        r_DrawQuadsRecursive(v1, t1, lDepth - 1);
        
        r_DrawQuadsRecursive(t1, v2, lDepth - 1);
        
        t2[0] = t1[0];
        t2[1] = v1[1];
        t2[2] = v1[2];
        
        t3[0] = v2[0];
        t3[1] = t1[1];
        t3[2] = t1[2];
        
       r_DrawQuadsRecursive(t2, t3, lDepth - 1);
        
        t2[0] = v1[0];
        t2[1] = t1[1];
        t2[2] = t1[2];
        
        t3[0] = t1[0];
        t3[1] = v2[1];
        t3[2] = v2[2];

       r_DrawQuadsRecursive(t2, t3, lDepth - 1);
                   
     }
     else {
    	
    	//draw the quad
        
        glBegin(GL_QUADS);
        	glVertex3f(v1[0],v1[1], v1[2]);
        	glVertex3f(v1[0],v1[1], v2[2]);
        	glVertex3f(v2[0], v2[1], v2[2]);
        	glVertex3f(v2[0],v1[1], v1[2]);
    	glEnd();
        }
        
     } 

void cRenderer::r_DrawStacks() {

	sStack oStack;

	for (long lIndex=0; lIndex<oWorld->oStacks.size(); lIndex++) {
	
		oStack = oWorld->oStacks[lIndex];

		glPushMatrix();

		glTranslatef(oStack.vPosition[0], oStack.vPosition[1]+0.01f, oStack.vPosition[2]);     
		glRotatef(oStack.vRotation[0] * 90.0f, 0.0f, 1.0f, 0.0f);
		glScalef(0.33f, card_thickness/2, 0.5f);

		glBegin(GL_QUADS);
        	glVertex3f(-1.0f, 0.0, -1.0f);
        	glVertex3f(1.0f, 0.0f, -1.0f);
        	glVertex3f(1.0f, 0.0f, 1.0f);
        	glVertex3f(-1.0f, 0.0f, 1.0f);
    	glEnd();

		glPopMatrix();
		}
	}

void cRenderer::r_DrawRoom() {
     
     
     if (oWorld->lRoomDisplayList !=0) {
     glCallList(oWorld->lRoomDisplayList);                            
     
     //strcat(oText.cText, "\n\rOK");
     
     }
     else {
          
     oWorld->lRoomDisplayList = glGenLists(1);
     
     glNewList(oWorld->lRoomDisplayList, GL_COMPILE);
     
     glPushMatrix();
     
     glTranslatef(0, 4, 0);
     glScalef(20, 8, 20);
     

     vec_f v1, v2, v3, v4, v5, v6, v7, v8;
    
     v1[0] = -1;
     v1[1] = 1;
     v1[2] = 0;
    
     v2[0] = 1;
     v2[1] = 1;
     v2[2] = 0;

     v3[0] = -1;
     v3[1] = -1;
     v3[2] = 0;

     v4[0] = 1;
     v4[1] = -1;
     v4[2] = 0;
    
     v5[0] = -1;
     v5[1] = 0;
     v5[2] = 1;
    
     v6[0] = 1;
     v6[1] = 0;
     v6[2] = 1;

     v7[0] = -1;
     v7[1] = 0;
     v7[2] = -1;

     v8[0] = 1;
     v8[1] = 0;
     v8[2] = -1;

     oWorld->fTextureRepeat = 4;

    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials + 1]);

     //bottom
     glPushMatrix();
     glTranslatef(0, -1, 0);     
     glRotatef(-90, 1, 0, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();


    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials + 4]);

     //top
     glPushMatrix();
     glTranslatef(0, 1, 0);     
     glRotatef(90, 1, 0, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();


     //north
     glPushMatrix();
     glTranslatef(0, 0, -1);     
     //glRotatef(+90, 1, 0, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();

     //south
     glPushMatrix();
     glTranslatef(0, 0, 1);     
     glRotatef(180, 1, 0, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();


     //west
     glPushMatrix();
     glTranslatef(-1, 0, 0);     
     glRotatef(+90, 0, 1, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();

    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials + 3]);

     //east
     glPushMatrix();
     glTranslatef(1, 0, 0);     
     glRotatef(-90, 0, 1, 0);
     r_DrawTrianglesRecursive(v1, v3, v2, 8);
     r_DrawTrianglesRecursive(v4, v2, v3, 8);
     glPopMatrix();
     oWorld->fTextureRepeat = 1;


     /*glPushMatrix();
     glTranslatef(0, -1, 0);     
     r_DrawTrianglesRecursive(v5, v6, v7, 12);
     r_DrawTrianglesRecursive(v8, v7, v6, 12);
     glPopMatrix();

     glPushMatrix();
     glRotatef(180, 1, 0, 0);     
     glTranslatef(0, -1, 0);     
     r_DrawTrianglesRecursive(v5, v6, v7, 12);
     r_DrawTrianglesRecursive(v8, v7, v6, 12);
     glPopMatrix();


  /* //top
     r_DrawTrianglesRecursive(v1, v3, v2, 12);
     r_DrawTrianglesRecursive(v4, v2, v3, 12);
     
     //bottom
     r_DrawTrianglesRecursive(v5, v6, v7, 12);
     r_DrawTrianglesRecursive(v8, v7, v6, 12);
     
     //north
     r_DrawTrianglesRecursive(v1, v2, v5, 12);
     r_DrawTrianglesRecursive(v6, v5, v2, 12);
     
     //south
     r_DrawTrianglesRecursive(v3, v7, v4, 12);
     r_DrawTrianglesRecursive(v8, v4, v7, 12);
     
     //west
     r_DrawTrianglesRecursive(v1, v5, v3, 12);
     r_DrawTrianglesRecursive(v7, v3, v5, 12);
     
     //east
     r_DrawTrianglesRecursive(v2, v4, v6, 12);
     r_DrawTrianglesRecursive(v8, v6, v4, 12);*/
     
     glPopMatrix();
     
     glEndList();
     }
     }

void cRenderer::r_DrawTable()
{
    if (oWorld->RenderOptions.iTableType == 0) return;
    
	//Let's use a shader.  Passing 0 will turn OFF a shader.
	r_UseShader(s_table);

	//Bind a woody texture
	glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[t_materials]);
    
    if (oWorld->RenderOptions.iTableType == 2) {

        glPushMatrix();
        
        glScalef(5,0.2f,5);
        glRotatef(90,1,0,0);  
       
        gluDisk(guDisk, 0.0f, 1.0f, 128, 24);
        gluCylinder(guDisk, 1.0, 1.0, 1.0, 64, 2);

        glTranslatef(0.0f, 0.0f, 1.0f);
        gluDisk(guDisk, 0.0f, 1.0f, 64, 2);
        
        glPopMatrix();
        }
    else {

        vec_f vCorner[8];
        
         vCorner[0][0] = -1;
         vCorner[0][1] = 1;
         vCorner[0][2] = 0;
        
         vCorner[1][0] = 1;
         vCorner[1][1] = 1;
         vCorner[1][2] = 0;

         vCorner[2][0] = -1;
         vCorner[2][1] = -1;
         vCorner[2][2] = 0;

         vCorner[3][0] = 1;
         vCorner[3][1] = -1;
         vCorner[3][2] = 0;

         vCorner[4][0] = -1;
         vCorner[4][1] = 1;
         vCorner[4][2] = 1;
        
         vCorner[5][0] = 1;
         vCorner[5][1] = 1;
         vCorner[5][2] = 1;

         vCorner[6][0] = -1;
         vCorner[6][1] = -1;
         vCorner[6][2] = 1;

         vCorner[7][0] = 1;
         vCorner[7][1] = -1;
         vCorner[7][2] = 1;
        
        glPushMatrix();
        
        glScalef(5,0.2f,5);
        glRotatef(90,1,0,0);  
        
        oWorld->fTextureRepeat = 1.0;
        
        //top
        r_DrawTrianglesRecursive(vCorner[0], vCorner[1], vCorner[2], 12);
        r_DrawTrianglesRecursive(vCorner[3], vCorner[2], vCorner[1], 12);
        
        //north
        r_DrawTrianglesRecursive(vCorner[2], vCorner[3], vCorner[7], 2);
        r_DrawTrianglesRecursive(vCorner[7], vCorner[6], vCorner[2], 2);
        
        //west
        r_DrawTrianglesRecursive(vCorner[6], vCorner[2], vCorner[0], 2);
        r_DrawTrianglesRecursive(vCorner[0], vCorner[4], vCorner[6], 2);
        
        //south
        r_DrawTrianglesRecursive(vCorner[0], vCorner[1], vCorner[4], 2);
        r_DrawTrianglesRecursive(vCorner[5], vCorner[4], vCorner[1], 2);

        //east
        r_DrawTrianglesRecursive(vCorner[1], vCorner[3], vCorner[7], 2);
        r_DrawTrianglesRecursive(vCorner[7], vCorner[5], vCorner[1], 2);
        
        
        //bottom
        r_DrawTrianglesRecursive(vCorner[4], vCorner[5], vCorner[6], 1);
        r_DrawTrianglesRecursive(vCorner[7], vCorner[6], vCorner[5], 1);
        
        oWorld->fTextureRepeat = 1.0;

        glPopMatrix();
        }

	//Turn of the shader
	r_TurnOffShader();

	if (oWorld->RenderOptions.bDrawCardLines) {
		r_DrawStacks();
		}
}

void cRenderer::r_DrawCube()
{
	// Colorful Quad
	glBegin(GL_QUAD_STRIP);
	glColor3f(1,0,0);
	glVertex3f(-0.5f,-0.5f,-0.5f);
	glVertex3f( 0.5f,-0.5f,-0.5f);
	glColor3f(1,1,0);
	glVertex3f(-0.5f, 0.5f,-0.5f);
	glVertex3f( 0.5f, 0.5f,-0.5f);
	glColor3f(0,1,0);
	glVertex3f(-0.5f, 0.5f, 0.5f);
	glVertex3f( 0.5f, 0.5f, 0.5f);
	glColor3f(0,1,1);
	glVertex3f(-0.5f,-0.5f, 0.5f);
	glVertex3f( 0.5f,-0.5f, 0.5f);
	glColor3f(1,0,1);
	glVertex3f(-0.5f,-0.5f,-0.5f);
	glVertex3f( 0.5f,-0.5f,-0.5f);
	glEnd();

	glColor3f(0,0,1);
	glBegin(GL_QUADS);
	
	glNormal3f(-1, 0, 0);
    
    glVertex3f(-0.5f,-0.5f,-0.5f);
	glVertex3f(-0.5f, 0.5f,-0.5f);
	glVertex3f(-0.5f, 0.5f, 0.5f);
	glVertex3f(-0.5f,-0.5f, 0.5f);

	glNormal3f(1, 0, 0);

	glVertex3f( 0.5f,-0.5f,-0.5f);
	glVertex3f( 0.5f, 0.5f,-0.5f);
	glVertex3f( 0.5f, 0.5f, 0.5f);
	glVertex3f( 0.5f,-0.5f, 0.5f);
	glEnd();
}

void cRenderer::r_DrawText(sText &poText) {

	float x1=(float)poText.iX, y1=(float)poText.iY;
	float y;
	unsigned int i;
	float cx, cy;
	
	char* cText;		//characters to be printed
	int offset = 0;		//offset within texture font: char 0 = image 0
	GLfloat vColor[4];	//font color
	float fSize;		//font size
	float fCharacterWidth; //width of character;
	bool bBlend;	    //true if blending was already enabled;
	bool bTexture2D;	//true if texture was already enabled;
	bool bPositionLineH;

	long lTextWidth;	//the width of the text to be printed;
	float lTextRowWidth;	//the width of the text to be printed;
	long lTextHeight;	//the height of the text to be printed;
	long lNumberLines;	//the number of lines to be printed;

	long lRectWidth;	//the width of the text rectangle;
	long lRectHeight;	//the height of the text rectangle;
	
	if(poText.cText==NULL)
		return;

	//save OpenGL settings/stats
	bBlend =	 glIsEnabled(GL_BLEND);
	bTexture2D = glIsEnabled(GL_TEXTURE_2D);

	glPushMatrix();

	//initialise variables
	cText = poText.cText;
	bPositionLineH = true;

	vColor[0] = poText.vColor[0];
	vColor[1] = poText.vColor[1];
	vColor[2] = poText.vColor[2];
	vColor[3] = poText.fTransparancy;

	lRectWidth = poText.cTextRect.iRight - poText.cTextRect.iLeft;
	lRectHeight = poText.cTextRect.iBottom - poText.cTextRect.iTop;

	r_CalculateTextRect(poText, lTextWidth, lTextHeight, lNumberLines);

	if (poText.dStyle & styleSizeFit) {
		
		fSize = poText.fSize * (float)lRectWidth / (float)lTextWidth;
		
		if (fSize > ((float)lRectHeight / (float)lNumberLines)) {

			fSize = (float)lRectHeight / (float)lNumberLines;
			}
		}
	else {	
		fSize = poText.fSize;
		}

	if (poText.dStyle & styleAlignCenterV) {
		y1=(float)poText.cTextRect.iTop + (lRectHeight - lNumberLines*fSize) / 2;
		}


	//change the OpenGL states
	glColor4fv(vColor);

	glEnable(GL_TEXTURE_2D);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  vColor);
    glBindTexture(GL_TEXTURE_2D, oWorld->lTextures[oFonts[lCurrentFont].lTextureId]);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	//print the characters
	for(i=0;i<strlen(cText);i++) {

		//move to next line if this is an eol sign.
		while(cText[i]=='\n' || cText[i]=='\r') {

			bPositionLineH = true;
			y1+=fSize;
			i++;
			}

		//Position the text line in horizontal directions.
		if (bPositionLineH) {

			r_CalculateTextLineRect(cText+i, fSize, lTextRowWidth);

			x1=(float)poText.iX;
			if (poText.dStyle & styleAlignRight) {
				x1=(float)poText.cTextRect.iRight - lTextRowWidth;
				}
			if (poText.dStyle & styleAlignCenterH) {
				x1=(float)poText.cTextRect.iLeft + (lRectWidth - lTextRowWidth) / 2;
				}
			bPositionLineH = false;
			}
		
		fCharacterWidth = fSize * ((float)oFonts[lCurrentFont].iCharacterWidth[*(cText+i)] / (float)64);

		//determine if the character is within the text rectangle 
		//and if not so clip all characters.
		bool bVisible;
		float fDeltaX1, fDeltaX2, fDeltaY1, fDeltaY2;		
		
		fDeltaX1 = fDeltaY1 = 0.0f;
		fDeltaX2 = fDeltaY2 = 1.0f;

		if (poText.dStyle & styleClipped) {
			bVisible = r_CalculateTextClipping(poText, x1, y1, fCharacterWidth, fSize, fDeltaX1, fDeltaX2, fDeltaY1, fDeltaY2);
			}
		
		//place the quads for the character
		if (bVisible) {
			cx=(float)fmod((cText[i]+offset)/16.0f, 1);
			cy=((cText[i]-offset)/16)/16.0f;
		    
			y = y1;
			glBegin(GL_QUADS);
				glTexCoord2f(cx + ((fSize-fCharacterWidth)/(fSize*2))*0.0625 + fDeltaX1*0.0625f, cy + fDeltaY2*0.0600f + 0.0025f);	glVertex2f(x1 + fDeltaX1*fCharacterWidth, y + fSize*fDeltaY2);
				glTexCoord2f(cx + ((fSize-fCharacterWidth)/(fSize*2))*0.0625 + fDeltaX1*0.0625f, cy + fDeltaY1*0.0625f);				glVertex2f(x1 + fDeltaX1*fCharacterWidth, y + fSize*fDeltaY1);
				glTexCoord2f(cx - ((fSize-fCharacterWidth)/(fSize*2))*0.0625 + fDeltaX2*0.0625f, cy + fDeltaY1*0.0625f);				glVertex2f(x1 + fCharacterWidth*fDeltaX2, y + fSize*fDeltaY1);
				glTexCoord2f(cx - ((fSize-fCharacterWidth)/(fSize*2))*0.0625 + fDeltaX2*0.0625f, cy + fDeltaY2*0.0600f + 0.0025f);	glVertex2f(x1 + fCharacterWidth*fDeltaX2, y + fSize*fDeltaY2);
			glEnd();
			}

		x1+=fCharacterWidth;
		}

	//Restore OpenGL states
	if (!bBlend) glDisable(GL_BLEND);
	if (!bTexture2D) glDisable(GL_TEXTURE_2D);
	
	glPopMatrix();
    }


//This function calculates the clipping variables. It returns true when a 
//character is visible and sets the delta variables accordingly.
bool cRenderer::r_CalculateTextClipping(sText &poText, long x1, long y1, 
										float fCharacterWidth, float fSize, 
										float &pfDeltaX1, float &pfDeltaX2, 
										float &pfDeltaY1, float &pfDeltaY2) {

	bool bVisible = true;
	
	if (x1<poText.cTextRect.iLeft) {
		
		if (poText.cTextRect.iLeft - x1 > fCharacterWidth) {
			bVisible = false;
			}
		else {
			pfDeltaX1 = (poText.cTextRect.iLeft - x1) / (fCharacterWidth);
			}
		}

	if (y1<poText.cTextRect.iTop) {
		
		if (poText.cTextRect.iTop - y1 > fSize) {
			bVisible = false;
			}
		else {
			pfDeltaY1 = (poText.cTextRect.iTop - y1) / fSize;
			}
		}

	if (y1 + fSize > poText.cTextRect.iBottom) {
		
		if (y1 > poText.cTextRect.iBottom) {
			bVisible = false;
			}
		else {
			 pfDeltaY2 = (poText.cTextRect.iBottom - y1) / fSize;
			}
		}

	if (x1 + fCharacterWidth > poText.cTextRect.iRight) {
		
		if (x1 > poText.cTextRect.iRight) {
			bVisible = false;
			}
		else {
			pfDeltaX2 = (poText.cTextRect.iRight - x1) / fCharacterWidth;
			}
		}

	return bVisible;
	}

		
//This function calculate the size of the text. This is a helper function
//for the r_DrawText function.

void cRenderer::r_CalculateTextRect(sText &poText, long &plWidth, long &plHeight, 
									long &plNumberLines) {

	float lRowWidth;
	char* pcText;
	char* cWWLastPosition;
	bool bContinue;
	float lWWPartRowWidth;

	//initialise variables
	plWidth = 0;
	plHeight = poText.fSize;
	lRowWidth = 0;
	plNumberLines = 1;
	pcText = poText.cText;

	//iterate through text string
	while (*pcText!=0x00) {

		cWWLastPosition = pcText;
		pcText = r_CalculateTextLineRect(pcText, poText.fSize, lRowWidth);
	
		//check for word wrap
		if (poText.iX + lRowWidth > poText.cTextRect.iRight && poText.dStyle & styleWordWrap) {

			bContinue = true;

			while (pcText != cWWLastPosition && bContinue) {

				pcText--;

				if (*pcText==0x20) {
				
					r_CalculateTextLineRect(pcText, poText.fSize, lWWPartRowWidth);

					if (poText.iX + lRowWidth - lWWPartRowWidth <= poText.cTextRect.iRight) {

						pcText[0] = 0x0A;
						pcText++;
						lRowWidth -= lWWPartRowWidth;
						bContinue = false;
						}
					}
				}
			}

		if (plWidth < lRowWidth) plWidth = lRowWidth;
		
		if (*pcText!=0x00) {
			
			lRowWidth = 0;
			plHeight += poText.fSize;
			plNumberLines++;
			pcText++;
			}
		}
}

//This function calculate the size of a text line. This is a helper function
//for the r_DrawText function.

char* cRenderer::r_CalculateTextLineRect(char *pcText, float pfSize, float &pfWidth) {

	//initialise variables
	pfWidth = 0;

	//iterate through text string
	while (*pcText!=0x00 && *pcText!='\n' && *pcText!='\r') {

		pfWidth += pfSize * ((float)oFonts[lCurrentFont].iCharacterWidth[*pcText] / (float)64);
		pcText++;
		}

	return pcText;
}


//This function determines if a window/button or a card in the hand of Player 0 (you) or on a stack on the table has been
//selected/clicked upon. The return value plType has the following options:
// -1: no card selected
// 0 : a card of player 0 is selected
// 1 : the top card of a stack is selected
// 2 : a window is pressed
// 3 : a button is pressed
// The plIndex gives or the index of the card in the hand of the player 
// or the index of the stack or the index of the window or button.

void cRenderer::r_Select(long &plType, long &plIndex, long &plIndex2) {
     
    GLuint selectBuf[BUFSIZE];
    GLint hits;
    long lSelectedIndex;
	long lButtonId;
	long lType;
	bool bModal;
	long lRow;
	    
    lSelectedIndex = -1;
	lButtonId = -1;
	lType = -1;
	bModal = false;
	lRow = 10; // the row in which a selected card is situated

    glSelectBuffer (BUFSIZE, selectBuf);
     
    r_DrawScene(GL_SELECT);
     
    hits = glRenderMode (GL_RENDER);

    GLuint names, *ptr;
    GLuint nHits, zMinimum, zMaximum, iName;
    ptr = (GLuint *) selectBuf;
     
    for (long lHits=0; lHits<hits; lHits++) {
        
        nHits = *ptr;
        ptr++;
        
        zMinimum = *ptr;
        ptr++;
        
        zMaximum = *ptr;
        ptr++;
        
        for (long lNames=0; lNames<(long)nHits; lNames++) {
            
            iName = *ptr;
            ptr++;
            
			//check for selection of a button on a window
			for (long lIndex=oWorld->oWindows.size()-1; lIndex>=0; lIndex--) {
				
				if (oWorld->oWindows[lIndex].bModal) bModal = true;
				
				for (long lButton=oWorld->oWindows[lIndex].oButtons.size()-1; lButton>=0; lButton--) {

					if (iName == 200 + oWorld->oWindows[lIndex].lWindowId * 10 + lButton + 1) {

						lSelectedIndex = oWorld->oWindows[lIndex].lWindowId;
						lButtonId = oWorld->oWindows[lIndex].oButtons[lButton].lId;
						lType = 3;
						}
					}
				}
			
			if (!bModal) {
			
				
				//check for selection of card in hand player 0
				for (long lIndex=(long)oWorld->oPlayers[0].oCards.size()-1; lIndex>=0; lIndex--) {
	                
					if (oWorld->oPlayers[0].oCards[lIndex]->lId == iName) {
	                                              
						//We are looking for the highest card in the nearest row
						if (lRow > oWorld->oPlayers[0].oCards[lIndex]->lRow || 
							(lRow == oWorld->oPlayers[0].oCards[lIndex]->lRow && lSelectedIndex<lIndex)) { 
						
							lSelectedIndex = lIndex;
							lRow = oWorld->oPlayers[0].oCards[lIndex]->lRow;
							lType = 0; // we have selected a card
							}
						}                
					}

				//check for selection of card on a stack
				for (long lIndex=0; lIndex<(long)oWorld->oStacks.size() && lType==-1; lIndex++) {

					for (long lSubIndex=0; lSubIndex<(long)oWorld->oStacks[lIndex].lCards.size() && lType==-1 ; lSubIndex++) {

						if (oWorld->oStacks[lIndex].lCards[lSubIndex] == iName) {
							lSelectedIndex = lIndex; //Instead of the card the stack is selected
							lType = 1; // we have selected a stack
							}
						}
					}
				}
            }
        }
     
	 plType = lType;
	 plIndex = lSelectedIndex;
	 plIndex2 = lButtonId;
     }

//This is the function which is called by the window application to render the
//scene.
void cRenderer::r_RenderScene() {
     
    float fFps;

	//position the cards in our world.
	for (long lIndex=oWorld->oPlayers.size()-1; lIndex>=0; lIndex--) {
       r_PositionCardsInHand(oWorld->oPlayers[lIndex], oWorld->oDeck); 
       }
    r_PositionDeck();
    
    //calculate the time between the last frame and determine how fast (in frames
	//per second) we are running. Do the action to be able to draw the FPS on the
	//screen
	r_TimeFrame();
    
    fFps = r_CalculateFPS();
    
    sprintf(oText.cText, "FPS: %.1f",fFps);
    
    //Draw the scene
    r_DrawScene(GL_RENDER);
    
	//Increment a counter. This counter counts to 360 and starts over again. Every
	//frame the counter is incremented.
	lIncrement = (lIncrement + 1) % 360;
    }

//This function renders the scene in OpenGL. 

void cRenderer::r_DrawScene(GLenum mode) {
     
    GLint viewport[4];
    glDepthMask(1); 
    glEnable(GL_NORMALIZE);
    
    //initiation phase
    switch (mode) {
    
        case GL_SELECT: {
             
           glGetIntegerv (GL_VIEWPORT, viewport);
    
           glRenderMode (GL_SELECT);

           glInitNames();
           glPushName(0);
           break;
           }
             
        case GL_RENDER: {
             
          	glRenderMode (GL_RENDER);
              
            glEnable(GL_MULTISAMPLE_ARB);
          	glEnable(GL_LIGHTING);
          	//glEnable(GL_POLYGON_SMOOTH);

            glClearColor(0.0f, 0.0f, 0.0f, 0.0);
          	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
 
			glEnable(GL_DEPTH_TEST);
			//glEnable(GL_BLEND);
            glEnable(GL_TEXTURE_2D);
           
			glPolygonMode(GL_BACK, oWorld->RenderOptions.bRenderLines ? GL_LINE: GL_FILL);
			glPolygonMode(GL_FRONT, oWorld->RenderOptions.bRenderLines ? GL_LINE : GL_FILL);

			break;
           }
		}    


	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
           
    if (mode==GL_SELECT) {
		/* create 3x3 pixel picking region near cursor location */
		gluPickMatrix((GLdouble) oSystem->oMouseState.vMouseLastPosition[0], (GLdouble) (viewport[3] - oSystem->oMouseState.vMouseLastPosition[1]), 5.0, 5.0, viewport);
	}
    
	gluPerspective(50.0f, 1.2f*(float)oSystem->iGameWindowWidth/(float)oSystem->iGameWindowHeight, 1, 100.0f);
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

	//set the world's rotation and translation
	
	glTranslatef(0, 0, oWorld->fEyeDistance);

	glRotatef(oWorld->fEyeRotation, 1, 0, 0);
    glRotatef(oWorld->fRotateWorldXAxis, 1, 0, 0);
    glRotatef(oWorld->fRotateWorldYAxis, 0, 1, 0);
		    
	glTranslatef(0, 1, 0);

	//This part will draw the room and the ambient objects. These are objects
	//which can't be selected. So there is no need to test for them in hit test mode
	
	if (mode==GL_RENDER) {
    
		r_SetLighting();

        GLfloat no_mat[] = { 0.0, 0.0, 0.0, 1.0f };
        GLfloat mat_ambient[] = { 0.7f, 0.7f, 0.7f, 1.0f };
        GLfloat mat_ambient_color[] = { 0.8f, 0.8f, 0.2f, 1.0f };
        GLfloat mat_diffuse[] = { 0.1f, 0.5f, 0.8f, 1.0f };
        GLfloat mat_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
        GLfloat no_shininess[] = { 0.0 };
        GLfloat low_shininess[] = { 50.0f };
        GLfloat high_shininess[] = { 100.0f };
        GLfloat mat_emission[] = {0.3f, 0.2f, 0.2f, 0.0};

		GLfloat mat_amb_diff_ground[] = { 0.5f, 0.5f, 0.30f, 1.0f };
        GLfloat mat_emission0[] = {0.0, 0.0, 0.0, 1.0f};
   
        if (oWorld->RenderOptions.bRenderEnvironment) {
            GLfloat mat_amb_diff_background[] = { 0.1f, 0.1f, 0.1f, 1.0f };
            
            glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, oWorld->RenderOptions.bFullLight ? mat_ambient : mat_amb_diff_background);
            
            glMaterialfv(GL_FRONT, GL_EMISSION, mat_emission0);
            glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
            glMaterialfv(GL_FRONT, GL_SHININESS, high_shininess);

            oRoom->r_DrawMesh();
			r_DrawRoom();
            }

		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat_amb_diff_ground);
		glMaterialfv(GL_FRONT, GL_EMISSION, mat_emission0);
        glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
        glMaterialfv(GL_FRONT, GL_SHININESS, low_shininess);

        r_DrawTable();

 	//oMesh->r_DrawMesh();
       }

   r_DrawCards(oWorld->oDeck, mode);
   
	for (long lIndex=oWorld->oPlayers.size()-1; lIndex>=0; lIndex--) {
		r_DrawPlayerHand(oWorld->oPlayers[lIndex], oWorld->oDeck, mode, lIndex==0);
		}

	if (mode==GL_SELECT) {
       glLoadName(200);
       }

                             
	//draw the hud and the popup windows	
	glDisable(GL_LIGHTING);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix ();
	glLoadIdentity();

	if (mode==GL_SELECT) {
		//due to a change in the projection matrix, we need to set the
		//pick matrix again
		gluPickMatrix((GLdouble) oSystem->oMouseState.vMouseLastPosition[0], (GLdouble) (viewport[3] - oSystem->oMouseState.vMouseLastPosition[1]), 1.0, 1.0, viewport);
	}

	gluOrtho2D(0.0f, oSystem->iGameWindowWidth, oSystem->iGameWindowHeight, 0.0f);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix (); 
	glLoadIdentity();

	glDisable(GL_DEPTH_TEST); 
	glDepthMask(0); 

	r_DrawHud(mode);
	if (mode==GL_RENDER) r_DrawText(oText);

	glEnable(GL_DEPTH_TEST); 

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

    //clean up phase

    glFlush();

	theta += 1.0f;

    if (mode==GL_RENDER) {
 	       
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_LIGHT0);
		glDisable(GL_LIGHT1);
		glDisable(GL_LIGHT2);
		glDisable(GL_LIGHTING);
		glDisable(GL_MULTISAMPLE_ARB);
		glDisable(GL_DEPTH_TEST);
		}
	}
     

//this functions sets the lighting in the room.
void cRenderer::r_SetLighting() {

	//Set the lighting
	GLfloat light_ambient[] = { 0.0f, 0.0f, 0.0f, 1.0f };
	GLfloat light_diffuse[] = { 0.5f, 0.5f, 0.5f, 0.5f };
	GLfloat light_specular[] = { 0.7f, 0.7f, 0.7f, 1.0f };

	GLfloat light_position1[] = { 3.0f, 2.0f, 0.0f, 1.0f };
	GLfloat light_position2[] = { 0, 4.0f, 0.0f, 1.0f };
	GLfloat light_position3[] = { 0.0f, 3.0f, 0.0f, 0.0f };

	
	//light 0
	glPushMatrix();

	glColor3f(1,1,1);
	glRotatef(theta, 0, 0.5f, 0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
	glLightfv(GL_LIGHT0, GL_POSITION, light_position1);	

	glEnable(GL_LIGHT0);

	glTranslatef(3.0, 2.0, 0.0);
	//DrawSquare();
	glPopMatrix();

	//light 1
	GLfloat light1_ambient[] = { 0.2f, 0.2f, 0.2f, 1.0f };
	GLfloat light1_diffuse[] = { 1.0f, 1.0f, 1.0f, 1.0f };
	GLfloat light1_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
	GLfloat light1_position[] = { 0.0, 3.0f, 0.0, 1.0f };
	GLfloat spot_direction1[] = { 0.0, -1.0f, 0.0 };

	glLightfv(GL_LIGHT1, GL_AMBIENT, light1_ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, light1_diffuse);
	glLightfv(GL_LIGHT1, GL_SPECULAR, light1_specular);
	glLightfv(GL_LIGHT1, GL_POSITION, light1_position);

	glEnable(GL_LIGHT1);

	glPushMatrix();
	glTranslatef(0.0f, 3.0f, 0.0f);
	glScalef(0.1f, 0.1f, 0.1f);
	r_DrawCube();
	glPopMatrix();

	//full lights
	if (oWorld->RenderOptions.bFullLight) {
				
		glLightf(GL_LIGHT2, GL_CONSTANT_ATTENUATION, 0.1f);
		glLightf(GL_LIGHT2, GL_LINEAR_ATTENUATION, 0.0);
		glLightf(GL_LIGHT2, GL_QUADRATIC_ATTENUATION, 0.0);        

		glLightfv(GL_LIGHT2, GL_AMBIENT, light_specular);
		glLightfv(GL_LIGHT2, GL_DIFFUSE, light_diffuse);
		glLightfv(GL_LIGHT2, GL_SPECULAR, light_specular);
		glLightfv(GL_LIGHT2, GL_POSITION, light_position3);	

		glLightf (GL_LIGHT2, GL_SPOT_CUTOFF, 10.f);
		GLfloat spot_direction[] = {0, -1.0f, -0.2f };
		glLightfv(GL_LIGHT2, GL_SPOT_DIRECTION, spot_direction);
		glLightf (GL_LIGHT2, GL_SPOT_EXPONENT, 100.0);

		glEnable(GL_LIGHT2);
		}
}


void cRenderer::r_PositionDeck() {
     
   sCard* oCard;
   long lDeckHeight[2];
   
   lDeckHeight[0] = 0;
   lDeckHeight[1] = 0;
   
   for (long lIndex=0; lIndex<oWorld->oDeck.oCards.size(); lIndex++) {
   
       oCard = oWorld->oDeck.oCards[lIndex];
       
       if (oCard->bVisible && !oCard->bAnimated && oCard->lStack == 0) {
           
		   /*oCard->vRotation[0]=0;
           oCard->vRotation[1]=0;
           oCard->vRotation[2]=0;
		   oCard->vRotation[3]=pi;

           oCard->vPosition[0]=-0.5f + (lIndex / 54);
           oCard->vPosition[1]=lDeckHeight[lIndex / 54] * card_thickness;
           oCard->vPosition[2]=0;
           
           lDeckHeight[lIndex / 54]++; */
           
           }
       }
     }

//This functions sets the position and rotations of the cards in the hand
//of a player so that it looks as an arc. 
void cRenderer::r_PositionCardsInHand(sPlayer &poPlayers, sDeck &poDeck) {
     
    long lNumberCards;
	long lCardRows;
	long lCardRow;
	std::vector<long> lNumberCardsInRow;
	std::vector<float> fCardWidthInRow;
	long lCardsInRow;
         
    //do the position 
    lNumberCards = (long)poPlayers.oCards.size(); 
     
    float fAngleSpaceToDivide;
    float fAngleBetweenCards;
    float fDistanceToPlayersPosition;
    float fCurrentAngle;
    float fWidthCards;
	float fPositionX, fPositionY, fPositionZ;
	float fDeltaFromBase;

	sCard* oCard;

    fWidthCards = 0;
	lCardRows = 1;
	lCardRow = 0;
	if (lNumberCards>1) {
    
		//determine the total width of the cards. A card has a width of 100, space between two cards have
		//a width of 0 - 100.
		lCardsInRow = 0;
		for (long lIndex=0; lIndex<lNumberCards; lIndex++) {

			oCard = poPlayers.oCards[lIndex];
			lCardsInRow++;

			if (oCard->bFake) {

				fWidthCards += oCard->lWidth;
				}
			else {
				fWidthCards += 100;
				}

			if (fWidthCards>2000.0f) {

				//the width exceeds the treshold, add an extra row
				lNumberCardsInRow.push_back(lCardsInRow);
				fCardWidthInRow.push_back(fWidthCards);
				lCardsInRow=0;
				lCardRow++;
				lCardRows++;
				fWidthCards = 0;
				}
			}
		
		lNumberCardsInRow.push_back(lCardsInRow);
		fCardWidthInRow.push_back(fWidthCards);

		//Space out the cards, maximum space between cards is 15 degrees. Maximum range is 100 degrees for all the cards
		fAngleSpaceToDivide = 100.0f;
        fAngleBetweenCards = fAngleSpaceToDivide / (fWidthCards/100.0f);
         
        if (fAngleBetweenCards>15) {
                                   
            fAngleBetweenCards = 15.0f;
            fAngleSpaceToDivide = 15.0f * ((fWidthCards - 100.0f) / 100.0f);
            } 
        }
    else {
         
         //only one card to space out
         fAngleSpaceToDivide = 0.0f;
         fAngleBetweenCards = 0.1f;
         }     
     
	if (lCardRows>1) {
		fAngleBetweenCards = 5.0f;
		fWidthCards = 2000.0f;
		}

    fDistanceToPlayersPosition = sqrt (poPlayers.vPosition[0]*poPlayers.vPosition[0] + poPlayers.vPosition[1]*poPlayers.vPosition[1] + poPlayers.vPosition[2]*poPlayers.vPosition[2]); 
    fCurrentAngle = (-fAngleBetweenCards * (fWidthCards/200.0f) + fAngleBetweenCards/2) * pi / 180.0f;
	fAngleBetweenCards = fAngleBetweenCards * pi / 180.0f;

	
	long lIndex;
	lIndex = -1;

	for (long lRowIndex=0; lRowIndex<lNumberCardsInRow.size(); lRowIndex++) {

		fCurrentAngle = -fAngleBetweenCards * (fWidthCards/200.0f) + fAngleBetweenCards/2;

		for (long lCardIndex=0; lCardIndex<lNumberCardsInRow[lRowIndex]; lCardIndex++) {
	        
			lIndex++;
			oCard = poPlayers.oCards[lIndex];
			oCard->lRow = lRowIndex;
			
			//poDeck.oCards[poPlayers.lCards[lIndex]].bDrawSelected = poPlayers.lSelectedIndex == lIndex;
			//position
			
			//place the card its base on the table its plane
			fPositionX = (lCardIndex * card_thickness + fDistanceToPlayersPosition / 2 - lRowIndex * card_thickness * 20) * sin(poPlayers.fRotationPosition);
			fPositionY = 0;
			fPositionZ = (lCardIndex * card_thickness + fDistanceToPlayersPosition / 2 - lRowIndex * card_thickness * 20) * cos(poPlayers.fRotationPosition);
			
			//position the card its base along the normal vector of the hand
			fDeltaFromBase = (float)sin(fCurrentAngle) * (1.0f + lRowIndex * 0.2f) ; 
			
			fPositionX += fDeltaFromBase * sin(poPlayers.fRotationCard);
			fPositionZ += -fDeltaFromBase * cos(poPlayers.fRotationCard);

			//position the card in an arc. Corrections for height are necessary or else the cards will be placed
			//in a strange order.
			fDeltaFromBase = -0.8f + (float)cos(fCurrentAngle) * (1.0f + lRowIndex * 0.2f);
			
			fPositionX += -sin(poPlayers.fRotationPosition) * (fDeltaFromBase / tan(pi * 0.4f)); 
			fPositionY += fDeltaFromBase;
			fPositionZ += -cos(poPlayers.fRotationPosition) * (fDeltaFromBase / tan(pi * 0.4f));

			oCard = poPlayers.oCards[lIndex];
			oCard->vPosition[0] = fPositionX;
			oCard->vPosition[1] = fPositionY;
			oCard->vPosition[2] = fPositionZ;
	        
			//rotation
			oCard->vRotation[0] = poPlayers.fRotationPosition;
			oCard->vRotation[1] = pi * 0.4f; 
			oCard->vRotation[2] = -fCurrentAngle;
			oCard->vRotation[3] = 0;
			
			//translation
			oCard->fTranslation = (fDistanceToPlayersPosition / 2);

			if (!oCard->bFake) {
				fCurrentAngle += fAngleBetweenCards;
				}
			else {
			
				fCurrentAngle += (fAngleBetweenCards * oCard->lWidth) / 100.0f;
				}
			}
		}
	}

void cRenderer::r_TimeFrame() {
     
     DWORD dCurrentTick;
     
     dCurrentTick = oSystem->GetTickCount();
     
     dFrameTickCount.insert(dFrameTickCount.begin(), dCurrentTick);
     
     if (dFrameTickCount.size() > 200) {
                                
         dFrameTickCount.resize(200);
         }
     }
     
float cRenderer::r_CalculateFPS() {
      
      float fReturn;
      
      fReturn = 0;

      if (dFrameTickCount.size()>2) {
                  
           fReturn = (1000.0f * dFrameTickCount.size()) / (dFrameTickCount[0] - dFrameTickCount[dFrameTickCount.size()-1]) ;
      }
      
      return fReturn;
}


char* cRenderer::textFileRead(char *fn) {


	FILE *fp;
	char *content = NULL;

	int count=0;

	if (fn != NULL) {
		fp = fopen(fn,"rt");

		if (fp != NULL) {
      
      fseek(fp, 0, SEEK_END);
      count = ftell(fp);
      rewind(fp);

			if (count > 0) {
				content = (char *)malloc(sizeof(char) * (count+1));
				count = fread(content,sizeof(char),count,fp);
				content[count] = '\0';
			}
			fclose(fp);
		}
	}
	return content;
}


