/*
   
   CardGame3D Engine
   Copyright 2005 - Meusesoft
   
   Version 0.1: November 2005 - 
   
   
   
   Module CardGameEngine
   
   Contains the code for interacting between the renderer and the cardgame. The logic
   of the card games will be inside the cardgame engines   
   
*/

#define _CRT_RAND_S
#include "DataStructures.h"
#include "System.h"
#include "World.h"
#include "Animator.h"
#include "Renderer.h"
#include "CardGameEngine.h"
#include <stdio.h>
#include <string.h>
#include <math.h>

// Constructor en destructor

cCardGameEngine::cCardGameEngine(cSystem* poSystem, cWorld* poWorld, cAnimator* poAnimator) {
                                               
    oSystem = poSystem;
    oWorld = poWorld;  
	oAnimator = poAnimator;
}

cCardGameEngine::~cCardGameEngine() {
                                    
	c_Clean();                         
}

// Event handlers

// This function determines whether it is allowed (according to the rules
// of the card game to select the given card

void cCardGameEngine::onCardSelect(long plType, long plIndex) {

	long lDeckIndex;
	long lTopCardIndex;
	
	if (plIndex < 0 || plType < 0) return;

	 switch (plType) {

		 case 0: //Player
			lDeckIndex = oWorld->oPlayers[0].oCards[plIndex]->lId;          
			break;

		 case 1: // Stack
			
			if (oWorld->oStacks[plIndex].eClickAction != stackSelect) return;

			lTopCardIndex = (long)oWorld->oStacks[plIndex].lCards.size()-1;
			lDeckIndex = oWorld->oStacks[plIndex].lCards[lTopCardIndex];
			break;
	 }

	 if (oState == ePickCards) {
	 
		 if (!oWorld->oDeck.oCards[lDeckIndex]->bAnimated) {
			oWorld->oDeck.oCards[lDeckIndex]->bSelected = !oWorld->oDeck.oCards[lDeckIndex]->bSelected;
			}
		 }
}

//this function processes a double click on a stack
void cCardGameEngine::onStackDoubleClick(long plStackIndex) {

	long lSize;

	switch (oWorld->oStacks[plStackIndex].eDoubleClickAction) {

		case stackDeal: {

			c_InitRound();
			break;
			}

		case stackGetOneCard: {

			if (plStackIndex==0) {
				c_DoGetCardFromPot(lActivePlayer);
				}
			else {
				lSize = oWorld->oStacks[plStackIndex].lCards.size();
				c_DoGetCardFromStack(lActivePlayer, oWorld->oStacks[plStackIndex].lCards[lSize-1] , 0);
				}

			break;	
			}
		}
	}

void cCardGameEngine::c_DoGetCardFromPot(long plPlayer) {

	long lSize;
	long lCardId;
	bool bContinue;
	sCard* oCard;
	
			long lTeam;
			bool bFound;
			long lDestinationStack;

			lTeam = plPlayer % (long)oTeams.size();
	
			//find a stack for the red 3's. Either it already exists or take an empty
			//one
			bFound = false;
			lDestinationStack = -1;

			for (long lIndex=0; lIndex<((long)oTeams[lTeam].oStacks.size()-1) && !bFound; lIndex++) {

				if (oWorld->oStacks[oTeams[lTeam].oStacks[lIndex]].lCards.size()==0) {
					if (lDestinationStack==-1) {

						lDestinationStack = oTeams[lTeam].oStacks[lIndex];	
						}
					}

				if (c_GetStackType(oTeams[lTeam].oStacks[lIndex]) == card3) {

					lDestinationStack = oTeams[lTeam].oStacks[lIndex];
					bFound = true;
					}
				}

	lSize = oWorld->oStacks[0].lCards.size();
	bContinue = true;

	while (lSize>=0 && bContinue) {

		oCard = oWorld->oDeck.oCards[oWorld->oStacks[0].lCards[lSize-1]];

		if (oCard->eType == card3 && (oCard->eColor==cardHearts || oCard->eColor==cardDiamonds)) {

			//the card is a red 3, send it to the stack with 3's.			
			c_DoCardFromToStack(lActivePlayer, 0, lDestinationStack, oCard->lId, 150);
			
			lSize--;
			}
		else {

			c_DoGetCardFromStack(lActivePlayer, oCard->lId , 150);
			bContinue = false;
			}
		}
	}


void cCardGameEngine::onWindowButton(long plWindowId, long plButtonId) {

	switch (plButtonId) {

		case btnMeld: {

			c_DoTurnAction(eTurnMeld);
			break;
			}

		case btnDiscard: {

			//WindowDestroy(plWindowId);
			c_DoTurnAction(eTurnDiscard);
			break;
			}
		}
	}

//this function is called on the start of a turn
void cCardGameEngine::onTurn() {

	switch (oWorld->oPlayers[lActivePlayer].oType) {
		
		case playerHuman: {

			if (oState==eInitialiseTurn) c_InitTurn();
			if (oState==ePickCards) c_CheckCanasta(lActivePlayer);
			if (oState==eFinishTurn) c_DoPassTurn();
			break;
			}
		
		case playerAI: {

			c_DoAITurn();
			break;
			}

		case playerNetwork: {


			break;
			}
		};
	}

// This function sorts the card in the hand of the player

long cCardGameEngine::onCardSort(long plPlayer, long plCard) {

	long lResult;
	long lValuePlayerCard, lValueCard;
	sCard* oCard;
	sCard* oPlayersCard;
	int iPoint;
	
	if (plPlayer<0 && plCard<0) return 0;

	oCard = oWorld->oDeck.oCards[plCard];
	lResult = 0;

	for (long lIndex=0; lIndex<oWorld->oPlayers[plPlayer].oCards.size(); lIndex++) {

		oPlayersCard = oWorld->oPlayers[plPlayer].oCards[lIndex];

		//score both cards
		lValueCard = c_FeatureCard(oCard) ? 8 : 0;
		lValueCard += (c_CardPoint(oPlayersCard) < c_CardPoint(oCard)) ? 4 : 0;
		lValueCard += (oPlayersCard->eType < oCard->eType) ? 2 : 0;
		lValueCard += (oPlayersCard->eColor < oCard->eColor) ? 1 : 0;

		lValuePlayerCard = c_FeatureCard(oPlayersCard) ? 8 : 0;
		lValuePlayerCard += (c_CardPoint(oPlayersCard) > c_CardPoint(oCard)) ? 4 : 0;
		lValuePlayerCard += (oPlayersCard->eType > oCard->eType) ? 2 : 0;
		lValuePlayerCard += (oPlayersCard->eColor > oCard->eColor) ? 1 : 0;
		
		if (lValueCard > lValuePlayerCard) lResult = lIndex+1;
		}

	return lResult;	
}

void cCardGameEngine::c_HussleDeck() {

	long lSwapCard;
	sCard* oCard;
	unsigned int iRandom;

	//Hussle the deck
	for (long lIndex=oWorld->oStacks[0].lCards.size() * 3; lIndex>=0; lIndex--) {

		if (rand_s(&iRandom)==0) {

			iRandom = (unsigned int)(((double)iRandom * oWorld->oStacks[0].lCards.size()) / UINT_MAX);
			iRandom = min(iRandom, oWorld->oStacks[0].lCards.size()-1); 

			lSwapCard = oWorld->oStacks[0].lCards[0];
			oWorld->oStacks[0].lCards[0] = oWorld->oStacks[0].lCards[iRandom];
			oWorld->oStacks[0].lCards[iRandom] = lSwapCard;		
			}
		}

	//reposition the cards in the stack
	for (long lIndex=oWorld->oStacks[0].lCards.size()-1; lIndex>=0; lIndex--) {

		oCard = oWorld->oDeck.oCards[oWorld->oStacks[0].lCards[lIndex]];
		oCard->vPosition[1] = oWorld->oStacks[0].vPosition[1] + lIndex * card_thickness;
		}
	}

//Initialisation functions

void cCardGameEngine::c_InitTournament() {
     
     
     }
     
void cCardGameEngine::c_InitGame() {
     
   sCard* oCard;
   sStack oStack;
   sPlayer oPlayer;
   sTeam oTeam;
   //Dependent of the type of game a deck of cards is compiled
   
	//create players
	for (long lIndex=0; lIndex<4; lIndex++) {	
		
		oPlayer.oType = (lIndex==0 ? playerHuman : playerAI);
		oWorld->oPlayers.push_back(oPlayer);	
		}

	oWorld->w_PositionPlayers();

	//create teams
	for (long lIndex=0; lIndex<oWorld->oPlayers.size()/2; lIndex++) {

		oTeam.oCanastas.clear();
		oTeam.oStacks.clear();

		oTeams.push_back(oTeam);
		}

	//create a deal stack	
	oStack.bStandardFaceUp = false;
	oStack.vDelta[0]=0.0f;
	oStack.vDelta[1]=0.0f;
	oStack.vDelta[2]=0.0f;
	oStack.vPosition[0]=0.5f;
	oStack.vPosition[1]=0.0f;
	oStack.vPosition[2]=0.0f;
	oStack.vRotation[0]=0.0f;
	oStack.vRotation[1]=0.0f;
	oStack.vRotation[2]=0.0f;

	oStack.eClickAction = stackNothing; // no action
	oStack.eDoubleClickAction = stackDeal; // deal
	oWorld->oStacks.push_back(oStack);

	//create a pot
	oStack.bStandardFaceUp = true;
	oStack.vDelta[0]=0.0f;
	oStack.vDelta[1]=0.0f;
	oStack.vDelta[2]=0.0f;
	oStack.vPosition[0]=-0.5f;
	oStack.vPosition[1]=0.0f;
	oStack.vPosition[2]=0.0f;
	oStack.vRotation[0]=0.0f;
	oStack.vRotation[1]=0.0f;
	oStack.vRotation[2]=0.0f;
	oStack.eClickAction = stackNothing; // take a card
	oStack.eDoubleClickAction = stackError; // take a card
	oWorld->oStacks.push_back(oStack);

	//Create the deck
   for (long lIndex=0; lIndex<108; lIndex++) {
   
       oCard = new sCard;

	   oCard->lId = lIndex; //id

       oCard->iBackside=t_backsides + (lIndex / 54);
	   oCard->lTexture = t_cards + (lIndex % 54);
       oCard->eColor=c_CastCardColor((lIndex % 54) / 13);
       oCard->eType=c_CastCardType((lIndex % 54) % 13);
	   if (oCard->eColor == cardUndefined) {
			oCard->eType=cardJoker;
			}
       
       oCard->bVisible=true;
       oCard->bSelected=false;
	   oCard->bAnimated=false;
	   oCard->bFlipTexture=false;
	   oCard->bFake=false;
	   oCard->bSecret=false;
	   oCard->fTransparancy = 1.0f;
       oCard->fScale=1.0f;
	   oCard->fTranslation=0.0f;
	   oCard->lStack = 0;

		oCard->vPosition[0] = oWorld->oStacks[0].vPosition[0];
		oCard->vPosition[1] = oWorld->oStacks[0].vPosition[1] + oWorld->oStacks[0].lCards.size() * card_thickness;
		oCard->vPosition[2] = oWorld->oStacks[0].vPosition[2];
		oCard->vRotation[0] = 0;
		oCard->vRotation[1] = 0;
		oCard->vRotation[2] = 0;
		oCard->vRotation[3] = oWorld->oStacks[0].bStandardFaceUp ? 0 : pi;
		oCard->fTranslation = 0.0f;

   		oWorld->oStacks[0].lCards.push_back(oCard->lId);

		oWorld->oDeck.oCards.push_back(oCard);
   }
	
   //Hussle the deck and fill the stack on the table
   c_HussleDeck();

   //Create the possible stacks on the table
   float fAngle;
   float fCos;
   float fSin;
   
   float fStackPositions[8][3] = {
	   {0.0f, 0.0f, 1.2f},
	   {-0.7f, 0.0f, 2.2f},
	   {0.7f, 0.0f, 2.2f},
	   {-1.4f, 0.0f, 3.2f},
	   {0.0f, 0.0f, 3.2f},
	   {1.4f, 0.0f, 3.2f},
	   {-2.1f, 0.0f, 3.2f},
	   {2.1f, 0.0f, 3.2f}};

	for (long lTeamIndex = 0; lTeamIndex < oTeams.size(); lTeamIndex++) {
	   
		for (long lIndex = 0; lIndex < 8; lIndex++) {

			for (long lPlayerIndex = 0; lPlayerIndex<2; lPlayerIndex++) {
		
				fSin = sin(oWorld->oPlayers[lTeamIndex + lPlayerIndex * (long)oTeams.size()].fRotationPosition);
				fCos = cos(oWorld->oPlayers[lTeamIndex + lPlayerIndex * (long)oTeams.size()].fRotationPosition);
		
				oStack.bStandardFaceUp = true;
				oStack.vDelta[0]=0.1f * fSin;
				oStack.vDelta[1]=0.0f;
				oStack.vDelta[2]=0.1f * fCos;
				oStack.vPosition[0]=fStackPositions[lIndex][0] * fCos + fStackPositions[lIndex][2] * fSin;
				oStack.vPosition[1]=fStackPositions[lIndex][1];
				oStack.vPosition[2]=fStackPositions[lIndex][2] * fCos + fStackPositions[lIndex][0] * fSin;
				oStack.vRotation[0]=1.0f * fSin;
				oStack.vRotation[1]=0.0f;
				oStack.vRotation[2]=1.0f * fCos;
				oStack.eClickAction = stackNothing; 
				oStack.eDoubleClickAction = stackError; 

				oWorld->oStacks.push_back(oStack);

				//add the stack index to the team
				oTeams[lTeamIndex].oStacks.push_back((long)oWorld->oStacks.size()-1);
				}
			}
	   }
   
   lDealer = lActivePlayer = oWorld->oPlayers.size()-1;

	oWorld->oDeck.fTransparancy = 1.0f;
   }
     
//This function initiates a new round in the game in progress.
//It sets the animator to deal the cards, and sets the active player
void cCardGameEngine::c_InitRound() {
     
	sCard* oCard;
	sCard* oFakeCard;
	sCardAnimation oCardAnimation;
	long lCard;
	long lDelay;

	//Initialise data
	for (long lIndex = oTeams.size()-1; lIndex>=0; lIndex--) {

		oTeams[lIndex].oCanastas.clear();
		}

	for (long lIndex = oWorld->oStacks.size()-1; lIndex>=2; lIndex--) {

		//oWorld->oStacks[lIndex].vDelta[0]=0.0f;
		//oWorld->oStacks[lIndex].vDelta[1]=0.0f;
		//oWorld->oStacks[lIndex].vDelta[2]=0.1f;
		}


	//Clear all running animations on cards
	oAnimator->ClearCardAnimations();

    //Deal cards to players
    lCard = oWorld->oStacks[0].lCards.size()-1;
	lDelay = 0;
	
	for (long lCardIndex=0; lCardIndex<13; lCardIndex++) {    

		for (long lPlayerIndex=0; lPlayerIndex<oWorld->oPlayers.size(); lPlayerIndex++) {

			oCard = oWorld->oDeck.oCards[oWorld->oStacks[0].lCards[lCard]];
			lCard--;

			oCardAnimation.lCard = oCard->lId;
			oCardAnimation.lPlayer = lPlayerIndex;
			oCardAnimation.bSecret = (lPlayerIndex!=0);
			oCardAnimation.eAnimation = animStackToHand;
			oCardAnimation.lAnimationData = onCardSort(lPlayerIndex, oCard->lId); 

			oFakeCard = new sCard;
			oFakeCard->bFake = true;
			oFakeCard->bAnimated = false;
			oFakeCard->bSelected = false;
			oFakeCard->bVisible = true;
			oFakeCard->lId = oCard->lId;
			oFakeCard->eType = oCard->eType;
			oFakeCard->eColor = oCard->eColor;
			oFakeCard->bWiden = true;
			oFakeCard->lWidth = 100;

			oWorld->oPlayers[lPlayerIndex].oCards.insert(oWorld->oPlayers[lPlayerIndex].oCards.begin() + oCardAnimation.lAnimationData, oFakeCard);

			oCardAnimation.lDelay = lDelay;		

			lDelay = lDelay + 100;

			oAnimator->AddCardAnimation(oCardAnimation);         //oCard = oWorld->oDeck.oCards[lIndex*13+lIndex2];
			oAnimator->RegisterFakeCard(oFakeCard);
			//oWorld->oPlayers[lIndex].oCards.push_back(oCard);
			//oCard->bVisible = false;
			}
		}

			
	//move a card from the stack to the deck
	do {
		oCard = oWorld->oDeck.oCards[oWorld->oStacks[0].lCards[lCard]];
		lCard--;

		oCardAnimation.lCard = oCard->lId;
		oCardAnimation.lStack = 1;
		oCardAnimation.eAnimation = animStackToStack;
		oCardAnimation.lDelay = lDelay;
		oCardAnimation.vDestinationRotation[0] = 0.0f;

		if (c_FeatureCard(oCard)) {
			oCardAnimation.vDestinationRotation[0] = (float)0.5f * pi;
			}

		lDelay = lDelay + 100;

		oAnimator->AddCardAnimation(oCardAnimation);

		} while (c_FeatureCard(oCard) && lCard>0);


	//change the activeplayer and the state of the turn
	lDealer++;
	lDealer = lDealer % (long)oWorld->oPlayers.size();
	lActivePlayer = lDealer;
	oState = eInitialiseTurn;


	//wait for the animations are done for the next actions.
	oAnimator->SetWaitForCardAnimations();
}

//this function is called after the animation for dealing is done. It
//changes the state so that the active player can do an action.

void cCardGameEngine::c_InitTurn() {

	//Check for red 3's on the beginning of the turn
	c_InitTurnCheckRed3();

	//the player is allowed to get a card from the stack or to
	//take the pot
	oWorld->oStacks[0].eClickAction = stackNothing; 
	oWorld->oStacks[0].eDoubleClickAction = stackGetOneCard; 
	oWorld->oStacks[1].eClickAction = stackNothing;
	oWorld->oStacks[1].eDoubleClickAction = stackGetAllCards;

	//show the buttons to make a decision
	c_ShowPlayerActionWindow();

	//change the state of the turn
	oState = ePickCards;
	}

void cCardGameEngine::c_InitTurnCheckRed3() {

	//canasta specific: check if the player has a red 3 in its hand, if so place it on the table
	//and get a card from the stack
	long lIndex;
	long lGetCards;
	long lStackSize;
	long lTeam;
	bool bFound;
	long lDestinationStack;

	lIndex = (long)oWorld->oPlayers[lActivePlayer].oCards.size()-1;
	lStackSize = (long)oWorld->oStacks[0].lCards.size();
	lGetCards = 0;
	lTeam = lActivePlayer % (long)oTeams.size();
	
	//find a stack for the red 3's. Either it already exists or take an empty
	//one
	bFound = false;
	lDestinationStack = -1;

	for (long lIndex=0; (lIndex<(long)oTeams[lTeam].oStacks.size()-1) && !bFound; lIndex++) {

		if (oWorld->oStacks[oTeams[lTeam].oStacks[lIndex]].lCards.size()==0) {
			if (lDestinationStack==-1) {

				lDestinationStack = oTeams[lTeam].oStacks[lIndex];	
				}
			}

		if (c_GetStackType(oTeams[lTeam].oStacks[lIndex]) == card3) {

			lDestinationStack = oTeams[lTeam].oStacks[lIndex];
			bFound = true;
			}
		}


	//Check the card
	while (lIndex>=0 && oWorld->oPlayers[lActivePlayer].oCards[lIndex]->eType==card3
		&& (oWorld->oPlayers[lActivePlayer].oCards[lIndex]->eColor==cardHearts
		|| oWorld->oPlayers[lActivePlayer].oCards[lIndex]->eColor==cardDiamonds)) {

		//move the card to a stack on the table
		c_DoSendCardToStack(lActivePlayer, lDestinationStack, oWorld->oPlayers[lActivePlayer].oCards[lIndex]->lId, 150);
		
		lIndex--;
		lGetCards++;
		}

	//get cards from the stack for the dropped 3's
	while (lGetCards>0) {

		c_DoGetCardFromPot(lActivePlayer);
		lGetCards--;
		}
}

//this functions checks if there is a canasta on the table, and if so then it will
//rearrange the stack.

void cCardGameEngine::c_CheckCanasta(long plPlayer) {

	long lStackCount;
	long lStackIndex;
	long lTeamIndex;
	long lSize;
	bool bNewCanasta;
	bool bReordered;
	sCanasta oCanasta;
	sCard* oCard;
	sStack oStack;

	lTeamIndex  = plPlayer % (long)oTeams.size();
	

	for (long lStackCount = (long)oTeams[lTeamIndex].oStacks.size()-1; lStackCount>=0; lStackCount--) {
	
		lStackIndex = oTeams[lTeamIndex].oStacks[lStackCount];
		
		if (oWorld->oStacks[lStackIndex].lCards.size()>=7) {
			
			//this stack has 7 or more cards, this will count as a canasta
			
			//check if it is a new canasta, if so add it to the team
			bNewCanasta = true;		
		
			for (long lCanastaIndex=(long)oTeams[lTeamIndex].oCanastas.size()-1; lCanastaIndex>=0 && bNewCanasta; lCanastaIndex--) {

				if (oTeams[lTeamIndex].oCanastas[lCanastaIndex].lStackIndex == lStackIndex) {

					bNewCanasta = false;
					}
				}

			//this canasta is new
			if (bNewCanasta) {

				oCanasta.lStackIndex = lStackIndex;
				oCanasta.bClean = true;

				//determine if it is clean or not. A canasta is clean when it doesn't have
				//a 2 or a joker
				for (long lCardIndex=(long)oWorld->oStacks[lStackIndex].lCards.size()-1; lCardIndex>=0 && oCanasta.bClean; lCardIndex--) {

					oCard = oWorld->oDeck.oCards[oWorld->oStacks[lStackIndex].lCards[lCardIndex]];

					if (c_FeatureCard(oCard)) {

						oCanasta.bClean = false;
						}
					}
			
				//add the canasta to the team
				oTeams[lTeamIndex].oCanastas.push_back(oCanasta);
				}
			}
		}

	for (long lStackIndex = (long)oTeams[lTeamIndex].oCanastas.size()-1; lStackIndex>=0; lStackIndex--) {

		oCanasta = oTeams[lTeamIndex].oCanastas[lStackIndex];

		//reorder the stack so a red or a black card is on top;
		bReordered = false;
		for (long lCardIndex=(long)oWorld->oStacks[oCanasta.lStackIndex].lCards.size()-1; lCardIndex>=0 && !bReordered; lCardIndex--) {

			oCard = oWorld->oDeck.oCards[oWorld->oStacks[oCanasta.lStackIndex].lCards[lCardIndex]];

			if (!c_FeatureCard(oCard)) {

				if ((oCanasta.bClean && (oCard->eColor == cardDiamonds || oCard->eColor==cardHearts)) ||
					(!oCanasta.bClean && oCard->eColor != cardDiamonds && oCard->eColor!=cardHearts)) {
					
						//swap the card so the wanted card is at the top of the stack
						
						long lSwapCard = oWorld->oStacks[oCanasta.lStackIndex].lCards[lCardIndex];

						oWorld->oStacks[oCanasta.lStackIndex].lCards[lCardIndex] = oWorld->oStacks[oCanasta.lStackIndex].lCards[oWorld->oStacks[oCanasta.lStackIndex].lCards.size()-1];
						oWorld->oStacks[oCanasta.lStackIndex].lCards[oWorld->oStacks[oCanasta.lStackIndex].lCards.size()-1] = lSwapCard;

						bReordered = true;
					}
				}
			}
		
		//rearrange the stack so it looks like a canasta on the table
		oStack = oWorld->oStacks[oCanasta.lStackIndex];
		lSize = (long)oStack.lCards.size()-1;
		for (long lCardIndex=lSize; lCardIndex>=0; lCardIndex--) {
		
			oCard = oWorld->oDeck.oCards[oStack.lCards[lCardIndex]];

			oCard->vPosition[0] = oStack.vPosition[0];
			oCard->vPosition[1] = oStack.vPosition[1] + lCardIndex * card_thickness;
			oCard->vPosition[2] = oStack.vPosition[2];
			//oCard->vRotation[0] = 0.25 * pi;
			if (lCardIndex == lSize) oCard->vRotation[0] = (float)0.25 * pi;
			if (lCardIndex != lSize) oCard->vRotation[1] = (float)pi;

			oWorld->oStacks[oCanasta.lStackIndex].vDelta[0] = 0;
			oWorld->oStacks[oCanasta.lStackIndex].vDelta[1] = 0;
			oWorld->oStacks[oCanasta.lStackIndex].vDelta[2] = 0;
			}
		}
	}

//this function cleans up the datastructures.

void cCardGameEngine::c_Clean() {

	for (long lIndex=(long)oWorld->oPlayers.size()-1; lIndex>=0; lIndex--) { 
	
		oWorld->oPlayers[lIndex].oCards.clear();
		}
	oWorld->oPlayers.clear();

	for (long lIndex = 0; lIndex<(long)oWorld->oStacks.size(); lIndex++) {

		oWorld->oStacks[lIndex].lCards.clear();
		}
	oWorld->oStacks.clear();
    
	for (long lIndex = 0; lIndex<(long)oWorld->oDeck.oCards.size(); lIndex++) {
		delete oWorld->oDeck.oCards[lIndex];                        
		}
	oWorld->oDeck.oCards.clear();
	}

/* ----------------------------------------------------------- */

//This function is the base function for processing of actions by the players
//during their turn

void cCardGameEngine::c_DoTurnAction(eTurnAction peAction) {

	switch (peAction) {

		case eTurnDiscard: {

			c_DoTurnDiscard();
			break;
			}

		case eTurnMeld: {

			c_DoTurnMeld();
			break;
			}
		}
	}


//This functions moves the selected card(s) from the deck to the 
//pot (stack1). Also it gives the turn to the next player
void cCardGameEngine::c_DoTurnDiscard() {

	sCard* oCard;

	//make sure only one card is selected
	if (oWorld->GetNumberCardSelected(lActivePlayer)>1) {
	
		//do error
		return;
		}

	//set the animation
	oCard = oWorld->GetSelectedCard(lActivePlayer, 1);

	if (oCard==NULL) return;

	c_DoSendCardToStack(lActivePlayer, 1, oCard->lId, 0);
	oAnimator->SetWaitForCardAnimations();

	//unselect card
	oCard->bSelected = false;

	//change state
	oState = eFinishTurn;
}

//This functions melds the selected cards to stacks on the table
void cCardGameEngine::c_DoTurnMeld() {

	bool bPossibleMeld;
	bool bFound;
	sCard* oCard;
	long lMinStack;
	long lDelay;
	long lTeam;

	lDelay = 0;
	lTeam = lActivePlayer % (long)oTeams.size();

	//find the first stack without a card on it
	bFound = false;

	for (long lStackCount = 0; lStackCount<(long)oTeams[lTeam].oStacks.size() && !bFound; lStackCount++) {
	
		lMinStack = oTeams[lTeam].oStacks[lStackCount];
		
		bFound = (oWorld->oStacks[lMinStack].lCards.size()==0);
		}

	if (!bFound) return;
	
	//check if the suggested meld is possible
	bPossibleMeld = c_CheckMeld(lActivePlayer);

	if (bPossibleMeld) {

		//Walk through cards in the hand of the active player and walk through
		//meld helper to move cards to the stacks
		for (long lCardIndex=(long)oWorld->oPlayers[lActivePlayer].oCards.size()-1; lCardIndex>=0; lCardIndex--) {
		
			oCard = oWorld->oPlayers[lActivePlayer].oCards[lCardIndex];

			if (oCard->bSelected) {

				//find the card in the meld helper to determine to which stack the card
				//must go
				for (long lMeldIndex = (long)oCheckMeldHelper.size()-1; lMeldIndex>=0; lMeldIndex--) {
				
					for (long lIndex = (long)oCheckMeldHelper[lMeldIndex].lCardIndex.size()-1; lIndex>=0; lIndex--) {
				
						if (oCheckMeldHelper[lMeldIndex].lCardIndex[lIndex]==lCardIndex) {

							if (oCheckMeldHelper[lMeldIndex].lStackIndex == -1) {

								oCheckMeldHelper[lMeldIndex].lStackIndex = lMinStack;
								lMinStack++;
								}
							
							c_DoSendCardToStack(lActivePlayer, oCheckMeldHelper[lMeldIndex].lStackIndex, oCard->lId, 250);

							oCard->bSelected = false;
							}
						}
					}
				}
		}

		oAnimator->SetWaitForCardAnimations();

		}
	}

//This function checks if the selected cards in the player its hands are 'meldable'.
//It returns true if it is.
bool cCardGameEngine::c_CheckMeld(long plPlayer) {

	sMeld oMeld;
	sCard* oCard;
	eCardType eType;
	long  lStackIndex;
	long  lStackCount;
	long  lTeam;
	bool  bNewMeld;
	bool  bReturn;
	
	oCheckMeldHelper.clear();
	bReturn = true;

	//fill the structure with the info of the stacks on the table
	lTeam = plPlayer % oTeams.size();
	bNewMeld = true;

	for (long lStackCount = (long)oTeams[lTeam].oStacks.size()-1; lStackCount>=0; lStackCount--) {
	
		lStackIndex = oTeams[lTeam].oStacks[lStackCount];

		eType = c_GetStackType(lStackIndex);

		if (eType!=card3 && eType!=cardTypeUndefined) {

			oMeld.eType = eType;
			oMeld.lStackIndex = lStackIndex;
			oMeld.lCardsInStack = (long)oWorld->oStacks[lStackIndex].lCards.size();
			oMeld.bCanasta = (oMeld.lCardsInStack>=7);
			oMeld.lFeatureCardsInStack = 0;
			oMeld.lCardsInHand = 0;
			oMeld.lFeatureCardsInHand = 0;
			
			for (long lCardIndex = oMeld.lCardsInStack-1; lCardIndex>=0; lCardIndex--) {

				oCard = oWorld->oDeck.oCards[oWorld->oStacks[lStackIndex].lCards[lCardIndex]];

				if (c_FeatureCard(oCard)) oMeld.lFeatureCardsInStack++;
				}	

			oMeld.lCardsInStack -= oMeld.lFeatureCardsInStack;

			//Add the canasta stacks at the end of the vector.
			if (oMeld.bCanasta) {
				oCheckMeldHelper.push_back(oMeld);
				}
			else {
				oCheckMeldHelper.insert(oCheckMeldHelper.begin(), oMeld);			
				}
			}
		}


	//add the cards in the hand of the player
	for (long lCardIndex=0; (lCardIndex<oWorld->oPlayers[plPlayer].oCards.size() && bReturn); lCardIndex++) { 
	
		oCard = oWorld->oPlayers[plPlayer].oCards[lCardIndex];
		if (oCard->bSelected) {
		
			bNewMeld = true;

			if (c_FeatureCard(oCard)) {
			
				//try to fit the card in a meld that doesn't have 3 cards yet

				for (long lMeldIndex = (long)oCheckMeldHelper.size()-1; (lMeldIndex>=0 && bNewMeld); lMeldIndex--) {

					if (oCheckMeldHelper[lMeldIndex].lCardsInHand == 2 && oCheckMeldHelper[lMeldIndex].lFeatureCardsInHand == 0) {

						bNewMeld = false;
						oCheckMeldHelper[lMeldIndex].lFeatureCardsInHand++;	
						oCheckMeldHelper[lMeldIndex].lCardIndex.push_back(lCardIndex);
						}
					}
				}
			
			for (long lMeldIndex = oCheckMeldHelper.size()-1; (lMeldIndex>=0 && bNewMeld); lMeldIndex--) {

				if (oCheckMeldHelper[lMeldIndex].eType == oCard->eType) {

					oCheckMeldHelper[lMeldIndex].lCardsInHand++;
					oCheckMeldHelper[lMeldIndex].lCardIndex.push_back(lCardIndex);
					bNewMeld = false;
					}

				if (c_FeatureCard(oCard)) {

					//try to fit the 2 or joker in a stack
				
					if (oCheckMeldHelper[lMeldIndex].lCardsInStack + oCheckMeldHelper[lMeldIndex].lCardsInHand > oCheckMeldHelper[lMeldIndex].lFeatureCardsInStack + oCheckMeldHelper[lMeldIndex].lFeatureCardsInHand + 1) {
				
						oCheckMeldHelper[lMeldIndex].lFeatureCardsInHand++;
						oCheckMeldHelper[lMeldIndex].lCardIndex.push_back(lCardIndex);
						bNewMeld = false;
						}
					}
				}

			//we cannot meld the 2 or the joker, this meld is not possible
			if (c_FeatureCard(oCard) && bNewMeld) {

				bReturn = false;
				bNewMeld = false;
				}


			//the selected cards doesn't have a stack yet, we will have to create
			//a new meld for it.
			if (bNewMeld) {

				oMeld.bCanasta = false;
				oMeld.eType = oCard->eType;
				oMeld.lCardsInHand = 1;
				oMeld.lCardsInStack = 0;
				oMeld.lFeatureCardsInHand = 0;
				oMeld.lFeatureCardsInStack = 0;
				oMeld.lStackIndex = -1;
				oMeld.lCardIndex.push_back(lCardIndex);

				oCheckMeldHelper.insert(oCheckMeldHelper.begin(), oMeld);
				}
			}
		}


	//check if we can meld:
	//no series of cards with less than 3 three cards, even if we add possible 2's or Jokers.
	for (long lMeldIndex = (long)oCheckMeldHelper.size()-1; lMeldIndex>=0 && bReturn; lMeldIndex--) {

		oMeld = oCheckMeldHelper[lMeldIndex];

		if (oMeld.lCardsInHand + oMeld.lFeatureCardsInHand + oMeld.lFeatureCardsInStack + oMeld.lCardsInStack < 3) bReturn = false;
		}

	return bReturn;
}

eCardType cCardGameEngine::c_GetStackType(long plStackIndex) {

	eCardType eReturn;
	long lIndex;
	
	eReturn = cardTypeUndefined;

	if (oWorld->oStacks[plStackIndex].lCards.size()==0) return eReturn;

	for (long lIndex=(long)oWorld->oStacks[plStackIndex].lCards.size()-1; (lIndex>=0 && eReturn==cardTypeUndefined); lIndex--) {

		eReturn = oWorld->oDeck.oCards[oWorld->oStacks[plStackIndex].lCards[lIndex]]->eType;
		if (eReturn == card2 || eReturn == cardJoker) eReturn = cardTypeUndefined;
		}

	return eReturn;
	}

//This functions gives the turn to the next player
void cCardGameEngine::c_DoPassTurn() {

	//give turn to the next player
	lActivePlayer++;
	lActivePlayer = lActivePlayer % (long)oWorld->oPlayers.size();

	//change the state
	oState = eInitialiseTurn;

	//remove window
	oWorld->DeleteWindow(lActionWindow);

	//do the appropiate action
	onTurn();
	}

//This function sets the animator to move a card form a stack to the player's hand
void cCardGameEngine::c_DoGetCardFromStack(long plPlayer, long plCard, long plDelay) {

	sCard* oFakeCard;
	sCard* oCard;
	sCardAnimation oCardAnimation;
	long lMaxDelay;

	lMaxDelay = oAnimator->lGetMaxCardAnimationDelay();

	oCard = oWorld->oDeck.oCards[plCard];

	oCardAnimation.lCard = plCard;
	oCardAnimation.lPlayer = plPlayer;
	oCardAnimation.eAnimation = animStackToHand;
	oCardAnimation.lAnimationData = onCardSort(plPlayer, plCard); 
	oCardAnimation.bSecret = (plPlayer!=0);

	oFakeCard = new sCard;
	oFakeCard->bFake = true;
	oFakeCard->bAnimated = false;
	oFakeCard->bSelected = false;
	oFakeCard->bVisible = true;
	oFakeCard->lId = oCard->lId;
	oFakeCard->eType = oCard->eType;
	oFakeCard->eColor = oCard->eColor;
	oFakeCard->bWiden = true;
	oFakeCard->lWidth = 0;

	oWorld->oPlayers[plPlayer].oCards.insert(oWorld->oPlayers[plPlayer].oCards.begin() + oCardAnimation.lAnimationData, oFakeCard);

	oCardAnimation.lDelay = lMaxDelay + plDelay;		

	oAnimator->AddCardAnimation(oCardAnimation);
	oAnimator->RegisterFakeCard(oFakeCard);
	}

//This function sets the animator to move a card from the player's hand to a stack
void cCardGameEngine::c_DoSendCardToStack(long plPlayer, long plStack, long plCard, long plDelay) {

	sCardAnimation oCardAnimation;
	sCard* oCard;
	long lMaxDelay;
	FillMemory(&oCardAnimation, sizeof(oCardAnimation), 0x00);

	lMaxDelay = oAnimator->lGetMaxCardAnimationDelay();

	oCard = oWorld->oDeck.oCards[plCard];

	oCardAnimation.lCard = plCard;
	oCardAnimation.lPlayer = plPlayer;
	oCardAnimation.lStack = plStack;
	oCardAnimation.eAnimation = animHandToStack;
	oCardAnimation.lAnimationData = 0;
	oCardAnimation.lDelay = lMaxDelay + plDelay;
	oCardAnimation.vDestinationRotation[0] = oAnimator->CalculateShortestRotation(oCard->vRotation[0], oWorld->oStacks[plStack].vRotation[0]);

	//this is Canasta specific
	if (plStack==1) {

		if (c_FeatureCard(oCard)) {
			oCardAnimation.vDestinationRotation[0] = 0.5f * pi;
			}
		}

	//add the animation to the animator
	oAnimator->AddCardAnimation(oCardAnimation);
	}

//This function moves a card from one stack to another
void cCardGameEngine::c_DoCardFromToStack(long plPlayer, long plFromStack, long plToStack, long plCard, long plDelay) {

	sCardAnimation oCardAnimation;
	long lMaxDelay;
	FillMemory(&oCardAnimation, sizeof(oCardAnimation), 0x00);

	lMaxDelay = oAnimator->lGetMaxCardAnimationDelay();

	oCardAnimation.lCard = plCard;
	oCardAnimation.lPlayer = lActivePlayer;
	oCardAnimation.lStack = plToStack;
	oCardAnimation.eAnimation = animStackToStack;
	oCardAnimation.lAnimationData = plFromStack;
	oCardAnimation.lDelay = lMaxDelay + plDelay;
	oCardAnimation.vDestinationRotation[0] = 0.0f;

	//add the animation to the animator
	oAnimator->AddCardAnimation(oCardAnimation);
}

//this private functions determines if the given card is a feature card:
//a 2, a red 3 or a joker.
bool cCardGameEngine::c_FeatureCard(sCard* poCard) {

	return (poCard->eType == cardJoker || poCard->eType == card2 ||
			(poCard->eType == card3 && (poCard->eColor == cardHearts || poCard->eColor == cardDiamonds)));
	}

//this function determines how many points the card in the hand is worth
int cCardGameEngine::c_CardPoint(sCard* poCard) {

	int iReturn;

	iReturn = 0;

	//if (poCard->bFake) return 0; //fake cards are worthless

	switch (poCard->eType) {

		case card2:
		case cardAce:
			iReturn = 20;
			break;

		case card3:
			if (poCard->eColor==cardHearts || poCard->eColor==cardDiamonds) {
				iReturn = 100;
				}
			else {
				iReturn = 5;
				}
			break;

		case card4:
		case card5:
		case card6:
		case card7:
			iReturn = 5;
			break;

		case card8:
		case card9:
		case card10:
		case cardJack:
		case cardQueen:
		case cardKing:
			iReturn = 10;
			break;

		case cardJoker:
			iReturn = 50;
			break;
		}

	return iReturn;
	}

//this is a private function which casts/translate the integer
//value to a card color
eCardColor cCardGameEngine::c_CastCardColor(int piColor) {

	switch (piColor) {
	
		case 0: return cardHearts;
		case 1: return cardDiamonds;
		case 2: return cardSpades;
		case 3: return cardClubs;
		default: return cardUndefined;
		}
	}

//this is a private function which casts/translate the integer
//value to a card type
eCardType cCardGameEngine::c_CastCardType(int piType) {

	switch (piType) {

		case 0: return card2;
		case 1: return card3;
		case 2: return card4;
		case 3: return card5;
		case 4: return card6;
		case 5: return card7;
		case 6: return card8;
		case 7: return card9;
		case 8: return card10;
		case 9: return cardJack;
		case 10: return cardQueen;
		case 11: return cardKing;
		case 12: return cardAce;
		default: return cardTypeUndefined;
	}
}

//this function makes the decisions for the AI player
void cCardGameEngine::c_DoAITurn() {

	switch (oState) {

		case eInitialiseTurn:

			c_InitTurnCheckRed3();

		case ePickCards:

			c_DoAITurnPickCard();
			break;
		
		case eDiscardCards:

			c_DoAITurnMeldOrDiscard();
			break;

		case eFinishTurn:

			c_DoPassTurn();
			break;
		}
	}

//this function determines if the AI player will take the pot or get a card
//from the stack
void cCardGameEngine::c_DoAITurnPickCard() {

	onStackDoubleClick(0);
	oAnimator->SetWaitForCardAnimations();
	
	oState = eDiscardCards;
	}

//this function determines if the AI player will meld some cards or discard a card
//and ends it turn
void cCardGameEngine::c_DoAITurnMeldOrDiscard() {

	oWorld->oPlayers[lActivePlayer].oCards[0]->bSelected = true;
	c_DoTurnAction(eTurnDiscard);
	}

//this functions creates the player action window on the screen. This window
//has the meld and discard button
void cCardGameEngine::c_ShowPlayerActionWindow() {

	sWindowAnimation oWindowAnimation;
	sWindow oWindow;

	float lDiscardLength;
	float lMeldLength;
	float lMaxLength;

	//let the renderer calculate the textsize
	//oRenderer->r_CalculateTextLineRect("Discard", wndButtonFontSize, lDiscardLength);
	//oRenderer->r_CalculateTextLineRect("Meld", wndButtonFontSize, lMeldLength);

	lMeldLength = 100.0;
	lDiscardLength = 100.0;

	lMaxLength = min(oSystem->iGameWindowWidth, lDiscardLength + lMeldLength + 40);
	
	//set the attributes of the window
	memset(&oWindow, 0, sizeof(oWindow));
	oWindow.bModal = false;
	oWindow.fTransparancy = 0.6f;
	oWindow.vBackgroundColor[0] = 0.0f;
	oWindow.vBackgroundColor[1] = 0.0f;
	oWindow.vBackgroundColor[2] = 0.0f;
	oWindow.vSize[0] = lMaxLength + 40;
	oWindow.vSize[1] = 50.0f;
	oWindow.vPosition[0] = 0;
	oWindow.vPosition[1] = 10;
	oWindow.eHorizontalPosition = posCenter;
	oWindow.eVerticalPosition = posBottom;
	oWindow.eHorizontalSize = sizeFixed;
	oWindow.eVerticalSize = sizeFixed;

	oWorld->AddButtonToWindow(oWindow, btnMeld, "Meld",
			10	, 10, 10 + lMeldLength + 20, 40);

	oWorld->AddButtonToWindow(oWindow, btnDiscard, "Discard",
			oWindow.vSize[0] - 30 - lDiscardLength, 10, oWindow.vSize[0] - 10, 40);

	oWindow.lWindowId = oWorld->AddWindow(oWindow);
	lActionWindow = oWindow.lWindowId;
	}