/*
   
   CardGame3D Engine
   Copyright 2005 - Meusesoft
   
   Version 0.1: November 2005 - 
   
   
   
   Module Animator
   
   Contains the code for processing the animations
   
*/

#include "DataStructures.h"
#include "System.h"
#include "World.h"
#include "Animator.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <sstream>

//Constructor and destructor

cAnimator::cAnimator(cSystem* poSystem, cWorld* poWorld) {

	oSystem = poSystem;
	oWorld = poWorld;

	bWaitForCardAnimations = false;
}

cAnimator::~cAnimator() {

	ClearCardAnimations();
	ClearWorldAnimations();
	ClearRegisteredFakeCards();
}

//Adding animations

void cAnimator::AddWorldAnimation(sWorldAnimation poWorldAnimation) {

	oWorldAnimations.push_back(poWorldAnimation);
}

void cAnimator::ClearWorldAnimations() {

	oWorldAnimations.clear();
}

void cAnimator::AddCardAnimation(sCardAnimation poCardAnimation) {

	poCardAnimation.lSequence = 0;
	
	oCardAnimations.push_back(poCardAnimation);

	DoAnimateCard(oCardAnimations.size()-1, 0);

}

long cAnimator::lGetMaxCardAnimationDelay() {

	long lMaxDelay;

	lMaxDelay = 0;

	for (long lIndex=oCardAnimations.size()-1; lIndex>=0; lIndex--) {
	
		lMaxDelay = max(oCardAnimations[lIndex].lDelay, lMaxDelay);
		}

	return lMaxDelay;
	}

void cAnimator::ClearCardAnimations() {

	oCardAnimations.clear();
	bWaitForCardAnimations = false;
}

//This function registers a fake card so it can be animated (changing width)
void cAnimator::RegisterFakeCard(sCard* poCard) {

	oFakeCards.push_back(poCard);
	}

//This function unregisters a fake card 
void cAnimator::UnregisterFakeCard(sCard* poCard) {

	bool bFound;
	long lIndex;

	bFound = false;
	lIndex = oFakeCards.size()-1;
	
	while (lIndex>=0 && !bFound) {

		if (oFakeCards[lIndex]==poCard) {

			oFakeCards.erase(oFakeCards.begin() + lIndex);
			}

		lIndex--;
		}
	}

//This function unregisters a fake card and removes it from the
//player its hand
void cAnimator::UnregisterAndDeleteFakeCard(sCard* poCard) {

	bool bContinue;

	UnregisterFakeCard(poCard);

	bContinue = true;

	for (long lPlayerIndex=oWorld->oPlayers.size()-1; lPlayerIndex>=0 && bContinue; lPlayerIndex--) {

		for (long lCardIndex=oWorld->oPlayers[lPlayerIndex].oCards.size()-1; lCardIndex>=0 && bContinue; lCardIndex--) {

			if (poCard==oWorld->oPlayers[lPlayerIndex].oCards[lCardIndex]) {

				oWorld->oPlayers[lPlayerIndex].oCards.erase(oWorld->oPlayers[lPlayerIndex].oCards.begin() + lCardIndex);
				delete poCard;

				bContinue = false;
				}
			}
		}	
	}

//This function deletes fake cards and clears the vector of the fake cards
void cAnimator::ClearRegisteredFakeCards() {

	for (long lIndex=0; lIndex<oFakeCards.size(); lIndex++) {

		delete oFakeCards[lIndex];
		}

	oFakeCards.clear();
	}	


//set the flag for waiting for the card animations by the card engine
void cAnimator::SetWaitForCardAnimations() {

	bWaitForCardAnimations = true;

	}

//this function returns true when there are no animations anymore and
//when the flag was set. It will return true only once.
bool cAnimator::CardAnimationsFinished() {

	bool bReturn;

	bReturn = (bWaitForCardAnimations && (oCardAnimations.size()==0));
	if (bReturn) bWaitForCardAnimations = false;

	return bReturn;
	}

void cAnimator::AddWindowAnimation(sWindowAnimation poWindowAnimation) {

	poWindowAnimation.lSequence = 0;
	poWindowAnimation.lAnimationData = 0;
	
	oWindowAnimations.push_back(poWindowAnimation);
}

void cAnimator::ClearWindowAnimations() {

	oWindowAnimations.clear();
}


//Processing animations

void cAnimator::Process(long lTimePassed) {

	if (oWorldAnimations.size()>0) DoAnimateWorld(lTimePassed);
	if (oCardAnimations.size()>0) DoAnimateCards(lTimePassed);
	if (oWindowAnimations.size()>0) DoAnimateWindows(lTimePassed);
}

//Animations
void cAnimator::DoAnimateWindows(long lTimePassed) {

	long lIndex = oWindowAnimations.size()-1;
	
	//Walk through all the 
	while (lIndex>=0) {

		DoAnimateWindow(lIndex, lTimePassed);

		//if the animation is complete, remove it.
		if (oWindowAnimations[lIndex].lAnimationData == oWindowAnimations[lIndex].lDuration) {

			oWindowAnimations.erase(oWindowAnimations.begin() + lIndex);
			}

		lIndex--;
		}
	}

void cAnimator::DoAnimateWindow(long plIndex, long plTimePassed) {

	long lWindowIndex;
	long lTimePassed;
	float fDivider;

	//do the delay;
	lTimePassed = plTimePassed;
	
	if (oWindowAnimations[plIndex].lDelay > 0) {

		if (plTimePassed>oWindowAnimations[plIndex].lDelay) {

			lTimePassed -= oWindowAnimations[plIndex].lDelay; 
			oWindowAnimations[plIndex].lDelay = 0;
			}
		else {
			
			oWindowAnimations[plIndex].lDelay -= lTimePassed;
			return;
			}
		}																							

	//find the window
	lWindowIndex = oWorld->FindWindowA(oWindowAnimations[plIndex].lWindowId);
	if (lWindowIndex==-1) return;

	//perform the animations
	oWindowAnimations[plIndex].lAnimationData = min(oWindowAnimations[plIndex].lAnimationData+lTimePassed, oWindowAnimations[plIndex].lDuration);

	fDivider = (float)oWindowAnimations[plIndex].lAnimationData / (float)oWindowAnimations[plIndex].lDuration; 
	
	if (oWindowAnimations[plIndex].lAnimation & AW_SIZE) {

		oWorld->oWindows[lWindowIndex].vSize[0] = oWindowAnimations[plIndex].vSizeStart[0] + fDivider * (oWindowAnimations[plIndex].vSizeEnd[0] - oWindowAnimations[plIndex].vSizeStart[0]);		
		oWorld->oWindows[lWindowIndex].vSize[1] = oWindowAnimations[plIndex].vSizeStart[1] + fDivider * (oWindowAnimations[plIndex].vSizeEnd[1] - oWindowAnimations[plIndex].vSizeStart[1]);		
	}

	if (oWindowAnimations[plIndex].lAnimation & AW_POSITION) {

		oWorld->oWindows[lWindowIndex].vPosition[0] = oWindowAnimations[plIndex].vPositionStart[0] + fDivider * (oWindowAnimations[plIndex].vPositionEnd[0] - oWindowAnimations[plIndex].vPositionStart[0]);		
		oWorld->oWindows[lWindowIndex].vPosition[1] = oWindowAnimations[plIndex].vPositionStart[1] + fDivider * (oWindowAnimations[plIndex].vPositionEnd[1] - oWindowAnimations[plIndex].vPositionStart[1]);		
	}

	if (oWindowAnimations[plIndex].lAnimation & AW_TRANSPARANCY) {

		oWorld->oWindows[lWindowIndex].fTransparancy = oWindowAnimations[plIndex].fTransparancyStart + fDivider * (oWindowAnimations[plIndex].fTransparancyEnd - oWindowAnimations[plIndex].fTransparancyStart);		
	
		for (long lTextIndex=0; lTextIndex<oWorld->oWindows[lWindowIndex].oTexts.size(); lTextIndex++) {

			oWorld->oWindows[lWindowIndex].oTexts[lTextIndex].fTransparancy = fDivider;
			}	
	}
}

void cAnimator::DoAnimateCards(long lTimePassed) {

	long lValue;
	sCard* oFakeCard;
	
	for (long lCardIndex = oFakeCards.size()-1; lCardIndex>=0; lCardIndex--) {

		oFakeCard = oFakeCards[lCardIndex];
			
		if (oFakeCard->bFake) {

			if (oFakeCard->bWiden) {

				oFakeCard->lWidth += lTimePassed / 4;
				if (oFakeCard->lWidth>100) {
					oFakeCard->lWidth = 100;
					}
				}
			else {

				oFakeCard->lWidth -= lTimePassed / 4;
				if (oFakeCard->lWidth<0) {

					UnregisterAndDeleteFakeCard(oFakeCard);
					}
				}
			}
		}

	for (long lIndex=oCardAnimations.size()-1; lIndex>=0; lIndex--) {

		DoAnimateCard(lIndex, lTimePassed);

		//remove animations from queue when they're done.
		if (oCardAnimations[lIndex].lSequence == -1) {

			oCardAnimations.erase(oCardAnimations.begin() + lIndex);
			}

	}
}

void	cAnimator::DoAnimateCard(long lAnimation, long lTimePassed) {

		//do animations
		switch (oCardAnimations[lAnimation].eAnimation) {

			case animHandToStack:
				DoAnimateCardHandToStack(lAnimation, lTimePassed);
				break;

			case animStackToHand:
				DoAnimateCardStackToHand(lAnimation, lTimePassed);
				break;

			case animStackToStack:
				DoAnimateCardStackToStack(lAnimation, lTimePassed);
				break;
		}
}

void cAnimator::DoAnimateCardStackToHand(long plAnimationIndex, long plTimePassed) {

	sCard* oCard;
	sCard* oFakeCard;
	sPlayer oPlayers;
	sCardAnimation oCardAnimation;
	long lPositionInHand;

	oCardAnimation = oCardAnimations[plAnimationIndex];

	oCard = oWorld->oDeck.oCards[oCardAnimation.lCard];
	oPlayers = oWorld->oPlayers[oCardAnimation.lPlayer];

	lPositionInHand = -1;
	for (long lIndex=0; lIndex<oPlayers.oCards.size(); lIndex++) {

		if (oPlayers.oCards[lIndex]->lId == oCard->lId) {
	
			oFakeCard = oPlayers.oCards[lIndex];
			lPositionInHand = lIndex;
			}
		}

	if (lPositionInHand==-1) {
		::MessageBox(NULL, "Unknow error occured while animating.", "Error", MB_OK | MB_ICONEXCLAMATION);	
		return; //error, fake card is gone
		}

	switch (oCardAnimation.lSequence) {

		case 0: { //remove card from stack

			//remove card from stack
			if (oCard->lStack==-1) {
				return;
				}
			
			oWorld->oStacks[oCard->lStack].lCards.erase(oWorld->oStacks[oCard->lStack].lCards.begin() + oWorld->oStacks[oCard->lStack].lCards.size()-1);
			oCard->lStack = -1;
			oCard->bSelected = false;
			oCard->bAnimated = true;
			oCard->bSecret = (oCardAnimation.lPlayer!=0);

			oCardAnimation.lSequence = 1;
			oCardAnimation.lAnimationData = 0;
			break;
			}

		case 1:{ //initialise animation

			//wait for delay to finish
			if (oCardAnimations[plAnimationIndex].lDelay > 0) {

				oCardAnimations[plAnimationIndex].lDelay -= plTimePassed;
				return;
				}

			//do some adjustment to the card on the stack it there is too much turning
			//involved
			if (abs(oCard->vRotation[0]-oFakeCard->vRotation[0])>0.5*pi) {
				
				oCard->vRotation[0] += pi;
				oCard->vRotation[0] = fmodf(oCard->vRotation[0], 2*pi);
				oCard->bFlipTexture = true;
				}

			//save the start position
			oCardAnimation.vStart[0] = oCard->vPosition[0];
			oCardAnimation.vStart[1] = oCard->vPosition[1]; 
			oCardAnimation.vStart[2] = oCard->vPosition[2];
			oCardAnimation.fStartTranslation = oCard->fTranslation;
			oCardAnimation.vStartRotation[0] = oCard->vRotation[0];
			oCardAnimation.vStartRotation[1] = oCard->vRotation[1];
			oCardAnimation.vStartRotation[2] = oCard->vRotation[2];
			oCardAnimation.vStartRotation[3] = oCard->vRotation[3];

			oCardAnimation.lSequence = 2;
			break;
			}

		case 2:{ //move to position above hand/fake card

			const long cThrowTime = 750;

			oCardAnimation.lAnimationData += plTimePassed;

			if (oCardAnimation.lAnimationData > cThrowTime) {
				oCardAnimation.lAnimationData = cThrowTime;
				oCardAnimation.lSequence++;
				}

			//calculate rotation vector of fakecard
			float fMatrix[4][4], fVector[4];

			glPushMatrix();
			glMatrixMode(GL_MODELVIEW);			
			glLoadIdentity();

			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			glRotatef(oFakeCard->vRotation[0] / pi * 180.0f, 0, 1, 0);
			glRotatef(oFakeCard->vRotation[1] / pi * 180.0f, 1, 0, 0);
			glRotatef(oFakeCard->vRotation[2] / pi * 180.0f, 0, 1, 0);
			glRotatef(oFakeCard->vRotation[3] / pi * 180.0f, 1, 0, 0);
			
			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			fVector[0]=0;
			fVector[1]=0;
			fVector[2]=-1;

			glPopMatrix();
	
			oCardAnimation.vAnimationVector[0] = fMatrix[0][0] * fVector[0] + fMatrix[1][0] * fVector[1] + fMatrix[2][0] * fVector[2];
			oCardAnimation.vAnimationVector[1] = fMatrix[0][1] * fVector[0] + fMatrix[1][1] * fVector[1] + fMatrix[2][1] * fVector[2];
			oCardAnimation.vAnimationVector[2] = fMatrix[0][2] * fVector[0] + fMatrix[1][2] * fVector[1] + fMatrix[2][2] * fVector[2];

			oCardAnimation.vDestination[0] = oFakeCard->vPosition[0] + oCardAnimation.vAnimationVector[0];
			oCardAnimation.vDestination[1] = oFakeCard->vPosition[1] + oCardAnimation.vAnimationVector[1];
			oCardAnimation.vDestination[2] = oFakeCard->vPosition[2] + oCardAnimation.vAnimationVector[2];

			//recalculate position of card
			oCard->vPosition[0] = oCardAnimation.vStart[0] + ((oCardAnimation.vDestination[0] - oCardAnimation.vStart[0]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[1] = oCardAnimation.vStart[1] + ((oCardAnimation.vDestination[1] - oCardAnimation.vStart[1]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[2] = oCardAnimation.vStart[2] + ((oCardAnimation.vDestination[2] - oCardAnimation.vStart[2]) / cThrowTime) * oCardAnimation.lAnimationData;

			oCard->vRotation[0] = oCardAnimation.vStartRotation[0] + ((oFakeCard->vRotation[0] - oCardAnimation.vStartRotation[0]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[1] = oCardAnimation.vStartRotation[1] + ((oFakeCard->vRotation[1] - oCardAnimation.vStartRotation[1]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[2] = oCardAnimation.vStartRotation[2] + ((oFakeCard->vRotation[2] - oCardAnimation.vStartRotation[2]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[3] = oCardAnimation.vStartRotation[3] + ((oFakeCard->vRotation[3] - oCardAnimation.vStartRotation[3]) / cThrowTime) * oCardAnimation.lAnimationData;

			oCard->fTranslation= oCardAnimation.fStartTranslation + ((oFakeCard->fTranslation - oCardAnimation.fStartTranslation) / cThrowTime) * oCardAnimation.lAnimationData;
			
			if (oCardAnimation.lSequence == 3) {

				oCardAnimation.lAnimationData = 0;
				}			
			break;
			}

		case 3: { //move card into hand

			const long cMoveTime = 500;

			oCardAnimation.lAnimationData += plTimePassed;

			if (oCardAnimation.lAnimationData > cMoveTime) {
				oCardAnimation.lAnimationData = cMoveTime;
				oCardAnimation.lSequence++;
				}

			//calculate rotation vector of fakecard
			float fMatrix[4][4], fVector[4];

			glPushMatrix();
			glMatrixMode(GL_MODELVIEW);			
			glLoadIdentity();

			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			glRotatef(oFakeCard->vRotation[0] / pi * 180.0f, 0, 1, 0);
			glRotatef(oFakeCard->vRotation[1] / pi * 180.0f, 1, 0, 0);
			glRotatef(oFakeCard->vRotation[2] / pi * 180.0f, 0, 1, 0);
			glRotatef(oFakeCard->vRotation[3] / pi * 180.0f, 1, 0, 0);
			
			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			fVector[0]=0;
			fVector[1]=0;
			fVector[2]=-1;

			glPopMatrix();
	
			oCardAnimation.vAnimationVector[0] = fMatrix[0][0] * fVector[0] + fMatrix[1][0] * fVector[1] + fMatrix[2][0] * fVector[2];
			oCardAnimation.vAnimationVector[1] = fMatrix[0][1] * fVector[0] + fMatrix[1][1] * fVector[1] + fMatrix[2][1] * fVector[2];
			oCardAnimation.vAnimationVector[2] = fMatrix[0][2] * fVector[0] + fMatrix[1][2] * fVector[1] + fMatrix[2][2] * fVector[2];

			//recalculate position of card
			oCard->vPosition[0] = oFakeCard->vPosition[0] + oCardAnimation.vAnimationVector[0] * (((float)cMoveTime - (float)oCardAnimation.lAnimationData) / (float)cMoveTime);
			oCard->vPosition[1] = oFakeCard->vPosition[1] + oCardAnimation.vAnimationVector[1] * (((float)cMoveTime - (float)oCardAnimation.lAnimationData) / (float)cMoveTime);
			oCard->vPosition[2] = oFakeCard->vPosition[2] + oCardAnimation.vAnimationVector[2] * (((float)cMoveTime - (float)oCardAnimation.lAnimationData) / (float)cMoveTime);

			oCard->vRotation[0] = oFakeCard->vRotation[0];
			oCard->vRotation[1] = oFakeCard->vRotation[1];
			oCard->vRotation[2] = oFakeCard->vRotation[2];
			oCard->vRotation[3] = oFakeCard->vRotation[3];

			oCard->fTranslation= oFakeCard->fTranslation;
			break;
			}


		case 4: { //remove animation from card

			oCard->bAnimated = false;
			oCard->bVisible = false;
			oCard->bSelected = false;
			oCard->bFlipTexture = false;

			//place card in hand of player
			oWorld->oPlayers[oCardAnimation.lPlayer].oCards[lPositionInHand] = oCard;
			UnregisterAndDeleteFakeCard(oFakeCard);
			
			oCardAnimation.lSequence = -1;
			break;
			}
	}
	
	oCardAnimations[plAnimationIndex] = oCardAnimation;
}
	
void cAnimator::DoAnimateCardHandToStack(long plIndex, long lTimePassed) {

	sCard* oCard;
	sCard* oFakeCard;
	sPlayer oPlayers;
	sCardAnimation oCardAnimation;
	long lIndex;


	if (oCardAnimations[plIndex].lDelay > 0) {

		oCardAnimations[plIndex].lDelay -= lTimePassed;
		return;
		}
	
	oCardAnimation = oCardAnimations[plIndex];
	oCard = oWorld->oDeck.oCards[oCardAnimation.lCard];
	oPlayers = oWorld->oPlayers[oCardAnimation.lPlayer];

	switch (oCardAnimation.lSequence) {

		case 0: { //init

			oCard->bVisible = true;
			oCard->bSelected = false;
			oCard->bAnimated = true;
			oCard->bSecret = (oCardAnimation.lPlayer!=0);

			oFakeCard = new sCard;
			oFakeCard->bFake = true;
			oFakeCard->bWiden = false;
			oFakeCard->lWidth = 100;
			oFakeCard->bSelected = false;
			oFakeCard->lId = oCard->lId;
			oFakeCard->eColor = oCard->eColor;
			oFakeCard->eType = oCard->eType;

			for (long lIndex = 0; lIndex<oWorld->oPlayers[oCardAnimation.lPlayer].oCards.size(); lIndex++) {

				if (oCard->lId == oWorld->oPlayers[oCardAnimation.lPlayer].oCards[lIndex]->lId) {

					oWorld->oPlayers[oCardAnimation.lPlayer].oCards[lIndex] = oFakeCard;
					}
				}

			RegisterFakeCard(oFakeCard);
			
			if (oCardAnimation.lStack >= 0 && oCardAnimation.lStack < oWorld->oStacks.size()) {
				
				oWorld->oStacks[oCardAnimation.lStack].lCards.push_back(oCardAnimation.lCard);
				oCardAnimation.vDestination[0] = oWorld->oStacks[oCardAnimation.lStack].vPosition[0] + oWorld->oStacks[oCardAnimation.lStack].vDelta[0] * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				oCardAnimation.vDestination[1] = oWorld->oStacks[oCardAnimation.lStack].vPosition[1] + oWorld->oStacks[oCardAnimation.lStack].vDelta[1] * oWorld->oStacks[oCardAnimation.lStack].lCards.size() + card_thickness * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				oCardAnimation.vDestination[2] = oWorld->oStacks[oCardAnimation.lStack].vPosition[2] + oWorld->oStacks[oCardAnimation.lStack].vDelta[2] * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				//oCardAnimation.vDestinationRotation[0] = 0.0f
				oCardAnimation.vDestinationRotation[1] = 0.0f;
				oCardAnimation.vDestinationRotation[2] = 0.0f;
				oCardAnimation.vDestinationRotation[3] = oWorld->oStacks[oCardAnimation.lStack].bStandardFaceUp ? 0 : pi;
				oCardAnimation.fDestinationTranslation = 0.0f;
				oCard->bSecret = ((oCardAnimation.lPlayer!=0) && !oWorld->oStacks[oCardAnimation.lStack].bStandardFaceUp);


				//do some adjustment to the card on the destination it there is too much turning
				//involved
				float fDeltaRotation;

				fDeltaRotation = abs(oCard->vRotation[0]-oCardAnimation.vDestinationRotation[0]);
				
				if (fDeltaRotation>0.5f*pi && abs(oCard->vRotation[0]-pi-oCardAnimation.vDestinationRotation[0])<fDeltaRotation) {
					
					oCardAnimation.vDestinationRotation[0] += pi;
					oCardAnimation.vDestinationRotation[0] = fmodf(oCardAnimation.vDestinationRotation[0], 2*pi);
					oCard->bFlipTexture = true;
					}
			}
			else {

				::MessageBox(NULL, "Card not connected to an existing stack", "Error", MB_OK);
			}

			float fMatrix[4][4], fVector[4];

			glPushMatrix();
			glMatrixMode(GL_MODELVIEW);			
			glLoadIdentity();

			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			glRotatef(oCard->vRotation[0] / pi * 180.0f, 0, 1, 0);
			glRotatef(oCard->vRotation[1] / pi * 180.0f, 1, 0, 0);
			glRotatef(oCard->vRotation[2] / pi * 180.0f, 0, 1, 0);
			glRotatef(oCard->vRotation[3] / pi * 180.0f, 1, 0, 0);
			
			glGetFloatv(GL_MODELVIEW_MATRIX, (float*)fMatrix); 

			fVector[0]=0;
			fVector[1]=0;
			fVector[2]=-1;

			glPopMatrix();
	
			oCardAnimation.lAnimationData = 0;
			oCardAnimation.vAnimationVector[0] = fMatrix[0][0] * fVector[0] + fMatrix[1][0] * fVector[1] + fMatrix[2][0] * fVector[2];
			oCardAnimation.vAnimationVector[1] = fMatrix[0][1] * fVector[0] + fMatrix[1][1] * fVector[1] + fMatrix[2][1] * fVector[2];
			oCardAnimation.vAnimationVector[2] = fMatrix[0][2] * fVector[0] + fMatrix[1][2] * fVector[1] + fMatrix[2][2] * fVector[2];

			oCardAnimation.lSequence = 1;
			break;
			}

		case 1: { //taking the card out of the hand

			oCardAnimation.lAnimationData = oCardAnimation.lAnimationData + lTimePassed;

			float fTotalTimePassed;
			fTotalTimePassed = oCardAnimation.lAnimationData;

			if (fTotalTimePassed>500) {

				oCardAnimation.lAnimationData = 0;
				oCardAnimation.lSequence = 2;
				
				lTimePassed -= fTotalTimePassed - 500;

				oCardAnimation.vStart[0] = oCard->vPosition[0];
				oCardAnimation.vStart[1] = oCard->vPosition[1]; 
				oCardAnimation.vStart[2] = oCard->vPosition[2];

				oCardAnimation.fStartTranslation = oCard->fTranslation;

				oCardAnimation.vStartRotation[0] = oCard->vRotation[0];
				oCardAnimation.vStartRotation[1] = oCard->vRotation[1];
				oCardAnimation.vStartRotation[2] = oCard->vRotation[2];
				oCardAnimation.vStartRotation[3] = oCard->vRotation[3];

			}
			
			oCard->vPosition[0] += (float)lTimePassed * (oCardAnimation.vAnimationVector[0] / 500.0f);
			oCard->vPosition[1] += (float)lTimePassed * (oCardAnimation.vAnimationVector[1] / 500.0f);
			oCard->vPosition[2] += (float)lTimePassed * (oCardAnimation.vAnimationVector[2] / 500.0f);

			float fTimeLeft;
			fTimeLeft = 500.0f - fTotalTimePassed;

			break;
		}

		case 2: { //move card to destination
			
			vec_f vDistance;
			float fLength;
			const long cThrowTime = 750;


			oCardAnimation.lAnimationData += lTimePassed;

			if (oCardAnimation.lAnimationData > cThrowTime) {
				oCardAnimation.lAnimationData = cThrowTime;
				oCardAnimation.lSequence = 3;
			}
			
			oCard->vPosition[0] = oCardAnimation.vStart[0] + ((oCardAnimation.vDestination[0] - oCardAnimation.vStart[0]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[1] = oCardAnimation.vStart[1] + ((oCardAnimation.vDestination[1] - oCardAnimation.vStart[1]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[2] = oCardAnimation.vStart[2] + ((oCardAnimation.vDestination[2] - oCardAnimation.vStart[2]) / cThrowTime) * oCardAnimation.lAnimationData;

			oCard->vRotation[0] = oCardAnimation.vStartRotation[0] + ((oCardAnimation.vDestinationRotation[0] - oCardAnimation.vStartRotation[0]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[1] = oCardAnimation.vStartRotation[1] + ((oCardAnimation.vDestinationRotation[1] - oCardAnimation.vStartRotation[1]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[2] = oCardAnimation.vStartRotation[2] + ((oCardAnimation.vDestinationRotation[2] - oCardAnimation.vStartRotation[2]) / cThrowTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[3] = oCardAnimation.vStartRotation[3] + ((oCardAnimation.vDestinationRotation[3] - oCardAnimation.vStartRotation[3]) / cThrowTime) * oCardAnimation.lAnimationData;

			oCard->fTranslation= oCardAnimation.fStartTranslation + ((oCardAnimation.fDestinationTranslation - oCardAnimation.fStartTranslation) / cThrowTime) * oCardAnimation.lAnimationData;

			break;
		}
		case 3: { //remove animation from card

			oCard->bAnimated = false;
			oCard->lStack = oCardAnimation.lStack;
			oCard->bSecret = false;

			if (abs(oCard->vRotation[0]>=pi)) {

				oCard->bFlipTexture = false;
				oCard->vRotation[0]-=pi;
				}

			oCardAnimation.lSequence = -1;
			break;

				}
	};

	oCardAnimations[plIndex] = oCardAnimation;
}

void cAnimator::DoAnimateCardStackToStack(long plIndex, long lTimePassed) {

	sCard* oCard;
	sPlayer oPlayers;
	sCardAnimation oCardAnimation;
	long lIndex;


	oCardAnimation = oCardAnimations[plIndex];
	oCard = oWorld->oDeck.oCards[oCardAnimation.lCard];

	switch (oCardAnimation.lSequence) {

		case 0: { //init

			oWorld->oStacks[oCard->lStack].lCards.erase(oWorld->oStacks[oCard->lStack].lCards.begin() + oWorld->oStacks[oCard->lStack].lCards.size()-1);
			oCard->lStack = -1;

			oCard->bVisible = true;
			oCard->bSelected = false;
			oCard->bAnimated = true;

			oCardAnimation.lAnimationData = 0;
			oCardAnimation.lSequence = 1;
			break;
			}

		case 1: { //init

			oCardAnimation.vStart[0] = oCard->vPosition[0];
			oCardAnimation.vStart[1] = oCard->vPosition[1]; 
			oCardAnimation.vStart[2] = oCard->vPosition[2];
			oCardAnimation.fStartTranslation = oCard->fTranslation;
			oCardAnimation.vStartRotation[0] = oCard->vRotation[0];
			oCardAnimation.vStartRotation[1] = oCard->vRotation[1];
			oCardAnimation.vStartRotation[2] = oCard->vRotation[2];
			oCardAnimation.vStartRotation[3] = oCard->vRotation[3];

			if (oCardAnimation.lStack >= 0 && oCardAnimation.lStack < oWorld->oStacks.size()) {
				
				oWorld->oStacks[oCardAnimation.lStack].lCards.push_back(oCardAnimation.lCard);
				oCardAnimation.vDestination[0] = oWorld->oStacks[oCardAnimation.lStack].vPosition[0] + oWorld->oStacks[oCardAnimation.lStack].vDelta[0] * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				oCardAnimation.vDestination[1] = oWorld->oStacks[oCardAnimation.lStack].vPosition[1] + oWorld->oStacks[oCardAnimation.lStack].vDelta[1] * oWorld->oStacks[oCardAnimation.lStack].lCards.size() + card_thickness * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				oCardAnimation.vDestination[2] = oWorld->oStacks[oCardAnimation.lStack].vPosition[2] + oWorld->oStacks[oCardAnimation.lStack].vDelta[2] * oWorld->oStacks[oCardAnimation.lStack].lCards.size();
				//oCardAnimation.vDestinationRotation[0] = 0.0f;
				oCardAnimation.vDestinationRotation[1] = 0.0f;
				oCardAnimation.vDestinationRotation[2] = 0.0f;
				oCardAnimation.vDestinationRotation[3] = oWorld->oStacks[oCardAnimation.lStack].bStandardFaceUp ? 0 : pi;
				oCardAnimation.fDestinationTranslation = 0.0f;
				oCardAnimation.bSecret = !oWorld->oStacks[oCardAnimation.lStack].bStandardFaceUp;
				oCard->bSecret = oCardAnimation.bSecret;
			}
			else {

				::MessageBox(NULL, "Card not connected to an existing stack", "Error", MB_OK);
			}

	
			oCardAnimation.lAnimationData = 0;

			oCardAnimation.lSequence = 2;
			break;
			}

		case 2: { //move card to destination
			
			if (oCardAnimations[plIndex].lDelay > 0) {

				oCardAnimations[plIndex].lDelay -= lTimePassed;
				return;
				}

			vec_f vDistance;
			float fLength;
			const long cMoveTime = 750;


			oCardAnimation.lAnimationData += lTimePassed;

			if (oCardAnimation.lAnimationData > cMoveTime) {
				oCardAnimation.lAnimationData = cMoveTime;
				oCardAnimation.lSequence = 3;
			}
			
			oCard->vPosition[0] = oCardAnimation.vStart[0] + ((oCardAnimation.vDestination[0] - oCardAnimation.vStart[0]) / cMoveTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[1] = oCardAnimation.vStart[1] + ((oCardAnimation.vDestination[1] - oCardAnimation.vStart[1]) / cMoveTime) * oCardAnimation.lAnimationData;
			oCard->vPosition[2] = oCardAnimation.vStart[2] + ((oCardAnimation.vDestination[2] - oCardAnimation.vStart[2]) / cMoveTime) * oCardAnimation.lAnimationData;

			oCard->vRotation[0] = oCardAnimation.vStartRotation[0] + ((oCardAnimation.vDestinationRotation[0] - oCardAnimation.vStartRotation[0]) / cMoveTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[1] = oCardAnimation.vStartRotation[1] + ((oCardAnimation.vDestinationRotation[1] - oCardAnimation.vStartRotation[1]) / cMoveTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[2] = oCardAnimation.vStartRotation[2] + ((oCardAnimation.vDestinationRotation[2] - oCardAnimation.vStartRotation[2]) / cMoveTime) * oCardAnimation.lAnimationData;
			oCard->vRotation[3] = oCardAnimation.vStartRotation[3] + ((oCardAnimation.vDestinationRotation[3] - oCardAnimation.vStartRotation[3]) / cMoveTime) * oCardAnimation.lAnimationData;

			oCard->vPosition[1] += sin((oCardAnimation.lAnimationData * pi)/ cMoveTime) / 1.4;
			
			oCard->fTranslation= oCardAnimation.fStartTranslation + ((oCardAnimation.fDestinationTranslation - oCardAnimation.fStartTranslation) / cMoveTime) * oCardAnimation.lAnimationData;

			break;
		}
		case 3: { //remove animation from card

			oCard->bAnimated = false;
			oCard->lStack = oCardAnimation.lStack;
			oCard->bSecret = false;
			
			oCardAnimation.lSequence = -1;
			break;

				}
	};

	oCardAnimations[plIndex] = oCardAnimation;
}

void cAnimator::DoAnimateWorld(long lTimePassed) {

    if (oWorld->fRotateWorldXAxis != 0) {
                                    
        oWorld->fRotateWorldXAxis = CalculateWorldRotation(lTimePassed, oWorld->fRotateWorldXAxis, oWorld->fVx);
        }
     
    if (oWorld->fRotateWorldYAxis != 0) {
                                    
        oWorld->fRotateWorldYAxis = CalculateWorldRotation(lTimePassed, oWorld->fRotateWorldYAxis, oWorld->fVy);
        }

	if (oWorld->fRotateWorldXAxis == 0 && oWorld->fRotateWorldYAxis == 0) {

		oWorldAnimations.clear();
		}
}

//This functions calculates what is the shortest rotation movement. For example
//if you want to rotate from 0 degrees to 270 degrees it would be shorter to rotate
// - 90.
// pFrom and pTo must be in the range of 0 to 2pi

float cAnimator::CalculateShortestRotation(float pFrom, float pTo) {

	float fTo;
	float fTo2;
	float fFrom;
	float fDelta;

	fTo = pTo;
	fFrom = pFrom;
	fDelta = pi - pFrom;

	fFrom += fDelta;
	fTo += fDelta;
	fTo2 = fTo - pi;

	if (abs(fFrom - fTo) > abs(fFrom - fTo2)) {
		
		fTo = fTo2;
		}	

	fTo -= fDelta;

	return fTo;
	}


float cAnimator::CalculateWorldRotation(float fTimePassed, float fRotation, float &fSpeed) {
      
      float fReturn;
      float fAcceleration;
      
      fReturn = fRotation;
      
      do {
      
          fAcceleration = fRotation / 5000; // the further from the origin the faster the acceleration;
          
          fSpeed += fAcceleration;
          
          if (fabs(fReturn)<=fabs(fSpeed)){
                        
             fSpeed = 0;
             return 0.0f;
             }
             
          fReturn -= fSpeed;
          
          fTimePassed--;
          
          } while (fTimePassed > 1);
      
      return fReturn;      
      } 